#!/bin/sh
#
# Script to create the fltk_config.make file.
#
#          (fltk must be installed before)
#
# The $1 command argument (when specified) is used to specify an FLTK version number.
#
# Set default options.
u_cc="gcc"
u_cxx="g++"
use_gl=
use_img=
use_frm=
use_glut=
use_cc=
use_cxx=
use_def=
use_debug=
use_jpg=
use_png=
use_spcimg=
use_libjpg=
use_libpng=
use_static=
opt_static=
opt_help=
opt_lvl="2"
opt_wall=
opt_ccoth=
opt_lnoth=
sys_env=
fout="fltk_config.make"
exe_ext=

if test $# -le 0
then
  opt_help=1
fi

# Loop on all argument to set the parameters.
while test $# -gt 0
do
  case "$1" in
    -cc | -CC)
      use_cc=$2
      shift
      ;;
    -cxx | -CXX | -c++ | -C++)
      use_cxx=$2
      shift
      ;;
    -gl | -GL)
      use_gl=" --use-gl"
      ;;
    -glut | -GLUT)
      use_glut=" --use-glut"
      ;;
    -img | -IMG)
      use_img=" --use-images"
      ;;
    -frm | -FRM | -forms | -FORMS)
      use_frm=" --use-forms"
      ;;
    -jpg | -jpeg | -JPG | -JPEG)
      use_img=" --use-images"
      use_jpg=1
      ;;
    -png | -PNG)
       use_img=" --use-images"
       use_png=1
       ;;
    -D*)
       use_def="${use_def} $1"
       ;;
    -g | -G)
      use_debug=" -g"
      opt_lvl=0
      ;;
    -o | -out | -OUT)
      fout=$2
      shift
      ;;
    -O | -opt | -OPT)
      opt_lvl = $2
      shift
      ;;
    -sta* | -STA*)
      opt_static=1
      ;;
    -ssta* | -SSTA*)
      opt_static=1
      use_static="-static "
      ;;
    -Wall | -WALL | -wall)
      opt_wall=" -Wall"
      ;;
    -ccsopt | -CCSOPT)
      opt_ccoth=" $2"
      shift
      ;;
    -lnsopt | -LNSOPT)
      opt_lnoth=" $2"
      shift
      ;;
    -h | -H | -help | -HELP)
      opt_help=1
      ;;
    *)
      echo " make_fltk_config.sh : Unknown option \"$1\""
      opt_help=1
    ;;
  esac
# Skip to next option/arg. in the command line.
  shift
done
#
if test "$use_debug" = " -g"
then
  opt_lvl=0
fi
#
if test $opt_help
# Output the Help message on unknown option.
then
  echo "Usage: make_fltk_config.sh [OPTIONS] 
Options:                                                                                   synonyms
    [-cc <cmp>]             use the specified C compiler (default is gcc),                  -CC
    [-cxx or -c++ <cmp>]    use the specified C++ compiler (default is g++)                 -CXX
    [-D<symbol>]            define the specified macro symbol (same that gcc option -D),
    [-gl]                   use GL or MESA-GL,                                              -GL
    [-glut]                 use glut compatibility layer,                                   -GLUT
    [-img]                  use extra images formats (PNG, JPEG),                           -IMG
    [-forms]                use forms compatibility layer,                                  -FORMS,  -frm,  -FRM
    [-jpg]                  use JPEG library,                                               -JPG,   -jpeg, -JPEG
    [-png]                  use P library,                                                  -PNG
    [-g]                    set debugging mode,                                             -G
    [-out <outfile_spc>]    Specify a not standard output,                                  -OUT, -o
    [-opt <lvl>]            Specify an optimize level (default: -opt 2),                    -OPT, -O
    [-sta]                  Specify use of static library (default no),                     -STA
    [-ssta]                 Specify use of static for system libraries (default no),        -SSTA
    [-wall]                 Specify option -Wall of compiler (default no),                  -Wall, -WALL
    [-ccsopt]               Specify inusual compiler option (default no),                   -CCSOPT
    [-lnsopt]               Specify inusual linker option (default no),                     -LNSOPT
    [-help]                 To output this text.                                            -HELP, -h, -H
"
  exit 1
fi

# Determine the operating system environment.
kernel=$(uname)
case $kernel in
  CYGWIN*)
    hom_win=" -DWINHOME"
    sys_env=" -D__WIN32 -D__CYGWIN__"
    kernel="Cygwin"
    exe_ext=".exe"
    ;;
  MINGW*)
    sys_env=" -D__WIN32"
    kernel="Mingw32"
    exe_ext=".exe"
    ;;
  Linux)
    sys_env=" -D__LINUX"
    kernel="Linux"
    ;;
  *)
    echo " Not identified Operating system \"${kernel}\" ==> Assume unknown UNIX."
    sys_env=" -D__UNIX"
    kernel="Unix"
    ;;
esac


# Generate the specific command strings.
fl_conf="fltk-config${use_debug}${use_gl}${use_glut}${use_img}${use_frm}"
echo "
 ***** Use the fltk config command \"$fl_conf\"     ******
"
#
#
if ! test $use_cc
then
  use_cc=$u_cc
fi

if ! test $use_cxx
then
  use_cxx=$u_cxx
fi

if test $use_png
then
#  Use a C program to test the png.h file existence.
  echo '
#include <stdio.h>
#include <libpng/png.h>
int main()
{
  printf( "test\n" );
}
' > test.c
  $use_cc -c test.c 2> /dev/null
  if test "$?" = "0"
  then
    use_png=
    rm test.o
  else
    use_png=" -DHAVE_PNG_H"
  fi
  rm test.c
fi

if ! test $opt_static
then
  ldflg_t="$($fl_conf --ldflags)${use_png}"
else
  ldflg_t="$use_static$($fl_conf --ldstaticflags)${use_png}"
fi
#
## echo " <$ldflg_t>"
# To write the sub-make file (we use the form $(command) in place of `command`.
echo "#
# Sub-Make File to define the FLTK compilation symbols (generated by \"make_fltk_config.sh\").
#
# Compiler specifications
CC  =   "${use_cc} -O${opt_lvl}${use_debug}${use_def}${sys_env}${opt_ccoth}${opt_wall}"
CXX =   "${use_cxx} -O${opt_lvl}${use_debug}${use_def}${sys_env}${opt_ccoth}${opt_wall}"

# Compiler flags specifications
CCFLG   =   "$($fl_conf --cflags) ${use_png}"
CXXFLG  =   "$($fl_conf --cxxflags) ${use_png}"

# linker flags (and libraries) specifications
LDFLG   =   ${ldflg_t}${opt_lnoth}
#
EXETYP  =   ${exe_ext}
#
" > $fout
#
# Create simple file to define manual compilation symbol CFLG and LDFLG.
#
echo "#
export CCFL=\""$($fl_conf --cflags) ${use_png}"\"
export CXFL=\""$($fl_conf --cxxflags) ${use_png}"\"
export LDFL=\""${ldflg_t}"\"
" > fltk_flags.sh
#
#

