//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module environement to manage the Slide Context Input/output Files
//
//


#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>


#include <FL/Fl.H>
#include <FL/fl_ask.H>
#include <FL/fl_utf8.h>         // To manipulate utf8 strings.


#include "DiaViewer_CTX.h"



int         CntxHdeIdx  =   0; // File version provided(for read), to use (for write) ...
                               // ... by CntxOpen() function.

static CntxHeader CntxKnowFrm[] = {
    "DVW   1.000-A00",
    "DVW   1.000-A01",
    ""
};

static CntxErrCatch_t   ErrHdl  = NULL; // Pointer to the user handler error routine.

static FilePath      * CntPath; // The Path used to scan the current volume.

static FILE          *   fcntx; // The comment file stream descriptor.
static const char *SPath= NULL; // The Current opened context file. 

static int              cindex; // The current index.


// To install the user error handler.
void CntxInsErrCatcher( CntxErrCatch_t hdl = NULL ) {  ErrHdl = hdl; }
// To Check if an Error handler is installed.
int  CntxInsErrCatcher() { return ErrHdl? 1 : 0; }



void CntxClose()
//
// This function must be used to close a slide context file.
//
{
    if (SPath) fclose( fcntx );
    SPath = NULL;
} // CntxClose().



static void IOError( int cod, const char *msg )
//
// IO Error routine.
//
// Routine use to generate all file context attached error message.
//
{
    if (cod > 0)
        fl_message( " * DiaViewer Warning # %d : %s\n * file\" %s\"", cod, msg, SPath );
    else {
        fl_alert( " * DiaViewer Error # %d : %s \n * file \"%s\"", -cod, msg, SPath );
        CntxClose();
        if (ErrHdl) ErrHdl( cod );      // Call user catcher to finish the current action.
    }
} // static void IOError( int cod, const char *msg ).



int CntxOpen( const char * fname, int wrt )
//
// This function must be used to open a slide context file.
// Return 1 on success, 0 on (read no such file), -1 on open error.
//
{
    CntxHeader    head, rdhd;
    char      head[16],
              rdhd[16];
    size_t         siz;
    int         ie = 1, // Assume no error.
              sz0, sz1;

    fcntx=fopen( fname, wrt?"w":"r" );
    if (!fcntx) {
        if (wrt) {     // Error : cannot create the new contextr file.
            fl_alert( " * DiaViewer Error: Cannot create context file \"%s\"\n * Error : %s",
                      fname, strerror( errno ) );
            ie = -1;
        } else {           
            if (errno !=ENOENT) { // if file exist with Open error, signal it.
                fl_message( " * DiaViewer Warning: Cannot Open context file \"%s\"\n * Error : %s\n",
                            fname, strerror( errno ) );
                ie = -1;
            } else ie = 0;
        }
    } else {
        SPath = fname;
        // The next string must be exactly 16 char (including the final \n (or \0)).
        sz0 = sprintf( head, "DVW %3d.%3.3u-%3s",
                       DIAVIEWER_VERSION, DIAVIEWER_SUB_VERSION, DIAVIEWER_RELEASE ) + 1;
        if (wrt)
            fprintf( fcntx, "%s\n", head );     // To Create/Supershed a new context file.
        else {
            sz1 = fscanf( fcntx,"%[^\n]",rdhd );// To read a Context file.
            while (
            
            if (strcmp( rdhd, head)) {
                fl_alert( " * DiaViewer Error: Bad format in context file \"%s\"\n", fname );
                CntxClose();
                ie = -1;
            }
        }
    }
    return ie;
} // CntxOpen( const char * fname, int wrt ).



//
// **** Routines to read context files of a volume.
//


char * CntxGetText( VarString &vs )
//
// Read any string of text..
//
{
    int   ie, siz, slu;
    char        frm[8];
    char         * str;

    ie = 1;                                     // Assume a sucess.
    slu = fscanf( fcntx, "%d", &siz );          // Get the size of string,
    if (slu<0) { siz = 0; ie = -1; }
    if (slu>0) {
        if (siz>0) {
            str = vs.Size( siz );               // Get temporary memory.
            sprintf( frm, " %%%dc", siz );      // Make the adapted read format.
            slu = fscanf( fcntx, frm, str );    // Read the text string.
            str[siz] = 0;                       // Append the final null character.
            if (slu <= 0) ie = -1;
        } else str = NULL;
    }
    if (ie<0) {
        str = NULL;
        if (ferror( fcntx )) IOError( -3, "Text Input context file error" );
                        else if (ErrHdl) ErrHdl( 1 );
    }
    return str;
} // char * CntxGetText().



int  CntxGetInt()
{
    int slu, val;

    slu = fscanf( fcntx, "%d", &val );
    if (slu<=0) {
        if (ferror( fcntx )) IOError( -7, "Integer Input context file error" );
                            else if (ErrHdl) ErrHdl( 1 );
    }
    return val;
} // CntxGetInt( short int &val ).



void CntxPutText( const char * str )
//
// Write a text.
//
{
    int len, swr;

    if (str&&str[0]) {
        len = strlen( str );
        swr = fprintf( fcntx, " %4d %s\n", len, str );
    } else swr = fprintf( fcntx, " %4d\n", 0 );
    if (swr<=0) {
        if (ferror( fcntx )) IOError( -8, "Text Output context file error" );
                        else if (ErrHdl) ErrHdl( 1 );
    }
} // int CntxPutText( const char * str ).



void CntxPutInt( int val, int fld )
//
// Write a text.
//
{
    int     swr;
    char frm[8];
    
    sprintf( frm, " %%%dd", fld );
    swr = fprintf( fcntx, frm, val );
    if (swr<=0) {
        if (ferror( fcntx )) IOError( -9, "Integer Output context file error" );
        else if (ErrHdl) ErrHdl( 1 );
    }
} // int CntxPutText( const char * str, int fld ).








//
// end of <PW-Id>.
//
