//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module environement to define and manage the Slide object
// by using the image manager for OpenGL ImageMan_REF.
//

#include <setjmp.h>

#include <FL/Fl.H>
#include <FL/fl_ask.H>

#include "DiaViewer_DIA.h"
#include "DiaViewer_PRG.h"
#include "Service_IO.h"



//
// Global symbols used for slides cache identifiers.
//

int IIdeOverflow  =                  0, // Flag for IIde_t overflow.
    SliCacheSize  = DEFAULT_CACHE_SIZE, // Size to use for Image Cache.
    IIdeFld[2]    =  UNIQUE_IDE_FIELDS, // Set default Fields.
    IIdeMaxSli    =                  0; // Maximum total number of slide/volume.

extern char *                 tvol_nam; // To get the ranking target volume name.


//
//  All local static variables.
//

static sigjmp_buf       JmpBuf; // Buffer for long jump on I/O context error.

static int     VolFlags =    0,
               VolPrgPc =    0,
               VolPrgId =    0;

static const char *  VolVname = NULL;

static VarString FnStr, // String Variable used as temporary strings.
                 TxStr;


// Context file supported format list :

static SrvFile::Header CntxFrm[] = {
    "DVW   1.001-A01", // # 1: ... The previous DiaViewer Version "DVW   1.001-A01 ...
    "DVW   1.000-A02", // # 2: ... this value has returned by the Open function ...
    "DVW   1.000-A01", // # 3: ... on success, 0 when no such file and -1 on ...
    "DVW   1.000-A00", // # 4: ... on error (unsupport format or I/O open/error.
    0
};

// Define the context file support (using Service_IO module).
static SrvFile CntxFile( CntxFrm, "Slide Context" );

static
    IIde_t  IIdeUniqueCnt =  0; // Count to define the Unique integer identifiers.

static int  ImageInitFlag =  1; // Flag for init Cache system.

// Define the structure of Unique identfiers.
static int  IIdeDirClear  =  0; // The slides ide. are unique in volume.
static int  IIdeVolClear  =  1; // The Slides and Dir. are not unique ...
                                // ... for all opened volumes in the run.

// It is implicite that the increment for slide is always 1.
static IIde_t    IIdeDirecIncr, // The increment for directory identifier field.
                 IIdeVolumIncr, // The increment for volume identifier field.
                 IIdeSlideMask, // The bit mask for the slide identifier field.
                 IIdeDirecMask, // The bit mask for the directory identifier field.
                 IIdeVolumMask; // The bit mask for the volume identifier field.



static
  DirRefer  * RefStack  = NULL; // Top of directory reference stack list.




// Routine Reference of GL module to check test for debug.
void OutMemsta( const char * s );


//
// Function to set the IIde increment and mask.
//

static void SetIIdeFields()
// At program init time it is possible to perform this
// function call to choice the IIde_t structure at run time :
//   SetIIdeFields( IIdeFld[0], IIdeFld[1] );
// If Bsli<0, the slide part of Ident is unique for the whole of each Volume.
// If Bdir<0, the directory part of the Ident is unique for all volume.
//
{
    int Bsli = abs( IIdeFld[0] ),
        Bdir = abs( IIdeFld[1] );

    IIdeDirClear = (IIdeFld[0]<0)?-1:0;
    IIdeVolClear = (IIdeFld[1]<0)?-1:0;
    IIdeDirecIncr = 1<<Bsli;
    IIdeVolumIncr = 1<<(Bsli + Bdir);
    IIdeSlideMask = IIdeDirecIncr-1;
    IIdeDirecMask = IIdeVolumIncr-IIdeDirecIncr;
    IIdeVolumMask = ~(IIdeVolumIncr-1);
    IIdeMaxSli    = IIdeVolumIncr-1;
}



//
// Functions for SliEntry class.
//

int  SliEntry::SetSliIIde()
{
    if ((IIdeUniqueCnt&IIdeSlideMask) < IIdeSlideMask) {
        Ident( ++IIdeUniqueCnt );
//printf( " Slide %10x name <%s>\n", Ident(), FName() );
        return 1;
    } else {
        IIdeOverflow    |=   1; // Flag a Slide identifier overflow.
        return 0;
    }
} // SliEntry::SetSliIIde().



SliEntry::SliEntry( const char * fname, IIde_t ide ) :
    Image_REF( fname, ide )
{
    if (ImageInitFlag) { // Force the Cache system init before the first slide allocation.
        SetIIdeFields();
        Image_Init( SliCacheSize );
        ImageInitFlag = 0;
    }
    
    stext_  = NULL;
    owner_  = NULL;
    uflgs_  =    0;
    slid_n_ = film_n_ = chap_n_ = note_n_ = 0;
} // SliEntry::SliEntry( const char * fname, IIde_t ide ).



SliEntry::SliEntry( const uchar * data, int ns, IIde_t ide, char * txt, char * uid ) :
    Image_REF( data, ide )
{
    if (ImageInitFlag) { // Force the Cache system init before the first slide allocation.
        SetIIdeFields();
        Image_Init( SliCacheSize );
        ImageInitFlag = 0;
    }
    
    owner_  = NULL;
    stext_  = txt;
    film_n_ = chap_n_ = note_n_ = 1; slid_n_ = ns;
} // SliEntry::SliEntry( const uchar * data, IIde_t = 0, int ns, char * txt ).



SliEntry::~SliEntry()
{
    if (stext_) free( stext_ );
} // SliEntry::~SliEntry().


void SliEntry::Rkg_SliProt( int pty, int flg )
//
// For a specified slide, set(flg=1) or Clear(flg=0) ...
// the slide protection bit flags specified by the mask pty.
//
{
    int fla = 0;
    if (pty&1) fla  = Context_ProInb;
    if (pty&2) fla |= Context_ProTxt;
    if (pty&4) fla |= Context_ProOri;
    if (pty&8) fla |= Context_LckRank;

    if (flg) FlgSet( fla );
        else FlgClr( fla );

} // void SliEntry::Rkg_SliProt( int pty, int flg ).





//
// Functions for DirEntry class.
//


int  DirEntry::SetDirIIde()
{
    if ((IIdeUniqueCnt&IIdeDirecMask) < IIdeDirecMask) {
        if (IIdeDirClear) IIdeUniqueCnt &= ~IIdeSlideMask;
        IIdeUniqueCnt +=  IIdeDirecIncr;
        ident_ = IIdeUniqueCnt&~IIdeSlideMask;
//printf( " Directory %10x name <%s>\n", ident_, fname_ );
        return 1;
    } else {
        IIdeOverflow    |=   2; // Flag a Directory identifier overflow.
        return 0;
    }
} // DirEntry::SetDirIIde().



DirEntry::DirEntry( const char *name, IIde_t ide )
{
    fname_ = strdup( name );  // The name is already copied to be keep in sort process.
    stext_ = NULL;
    ident_ =  ide;
    owner_ = NULL;
    dlst_  = NULL;
    slst_  = NULL;
    deep_  =    0;
    udeep_ =    0;
    ndir_  =    0;
    nsli_  =    0;
    cnsli_ =    0;
    flags_ =    0;
} // DirEntry::DirEntry( const char *name, IIde_t ide ).



DirEntry::~DirEntry()
{
    if (ndir_) {
        if (ndir_==1) ((DirEntry*)(dlst_))->~DirEntry();
    } else {
        for(int i=0;i<ndir_;i++) dlst_[i]->~DirEntry();
        delete[] dlst_;
    }
    if (nsli_) {
        if (nsli_==1) ((SliEntry*)(slst_))->~SliEntry();
        else {
            for(int i=0;i<nsli_;i++) slst_[i]->~SliEntry();
            delete[] slst_;
        }
    }
    if (fname_&&deep_)delete[] fname_;
    if (stext_) delete[] stext_;
    owner_ = NULL;
} // DirEntry::~DirEntry().



void DirEntry::SbDir( int nd, DirEntry **tb )
{
    if (nd <= 0) return;
    ndir_ = nd;
    if (nd > 1 ) {
        dlst_ = new DirEntry*[ nd ];
        for (int i=0; i<nd; i++) {
            tb[i]->owner_ = this;
            dlst_[i] = tb[i];
            tb[i] = NULL;
        }
    } else {
        tb[0]->owner_ = this;
        dlst_ = (DirEntry**)tb[0];
        tb[0] = NULL;
    }
    // The delete of table tb must be performed by caller.
} // void DirEntry::SbDir( int nd, DirEntry **tb ).



void DirEntry::Slide( int ns, SliEntry **tb )
{
    if (ns <= 0) return;
    nsli_ = ns;
    if (ns > 1) {
        slst_ = new SliEntry*[ ns ];
        for (int i=0; i<ns; i++) {
            tb[i]->owner_ = this;
            slst_[i] = tb[i];
            tb[i] = NULL;
        }
   } else {
        tb[0]->owner_ = this;
        slst_ = (SliEntry**)tb[0];
        tb[0] = NULL;
    }
    // The delete of table tb must be performed by caller.    
} // void DirEntry::Slide( int ns, SliEntry **tb ).



DirEntry * DirEntry::SbDir( int idx )
{
    if ((ndir_ <= 0)||(idx <= 0)||(idx > ndir_)) return NULL;
    return (ndir_ > 1)? dlst_[idx-1] : (DirEntry*)dlst_;
} // DirEntry * DirEntry::SbDir( int idx = 0 ).



SliEntry * DirEntry::Slide( int idx )
{
    if ((nsli_ <= 0)||(idx <= 0)||(idx > nsli_)) return NULL;
    return (nsli_ > 1)? slst_[idx-1] : (SliEntry*)slst_;    
} // SliEntry * DirEntry::Slide( int idx = 0 ).



void DirEntry::SetDirIdents()
{
    if (SetDirIIde()) {
        if (ndir_ > 0) {
            DirEntry ** tdir = dlst_;
            if (ndir_ > 1 ) {
                for (int ii=0;ii<ndir_;ii++) tdir[ii]->SetDirIdents();
            } else {
                ((DirEntry*)tdir)->SetDirIdents();
            }
        }
        if (nsli_ > 0) {
            SliEntry ** tsli = slst_;
            if (nsli_ > 1) {
                for (int ii=0;ii<nsli_;ii++) tsli[ii]->SetSliIIde();
            } else {
                ((SliEntry*)tsli)->SetSliIIde();
            }
        }
    }
} // DirEntry::SetDirIdents().



SliEntry * DirEntry::SliLocate( const char * fname )
{
    SliEntry * sli = NULL;
    
    if (nsli_) {
        if (nsli_ == 1) {
            sli = (SliEntry*)slst_;
            if (strcmp( fname, sli->FName() ) != 0) sli = NULL;
        } else {
            int ii = 0;
            while (ii < nsli_) {
                sli =slst_[ii++];
                if (strcmp( fname, sli->FName() ) == 0) break;
                sli = NULL;
            }
        }
    }
    return sli;
} // SliEntry * DirEntry::Locate( const char * fname ).



void SliEntry::SliCntxFlags()
//
// To save a slide context when it was modified.
//
{
    DirEntry * dir = owner_;

//OutMemsta( "SliCntxFlags  1" );
    if (FlgTst( Context_Update )) {
        dir->FlgSet( Context_Update );
        while (dir->LDeep()) {
            dir = dir->Owner();
            dir->FlgSet( Context_SubDir|Context_Update );
        }
        // Now deep is the pointer of Volume root directory.
        VolEntry * vol = (VolEntry*)(dir->Owner());
        vol->FlgSet( Context_Update );
    }
} // void SliEntry::ChkSliCntxFlags().



void SliEntry::RkgClear()
// To Clear the ranking data of one slide.
{
    if (!FlgTst( Context_ProInb )) {
        slid_n_ = film_n_ = chap_n_ = note_n_ = 0;
        FlgClr( Context_Slinbr );
        FlgSet( Context_Update );
    }
} // void SliEntry::RkgClear().



static const char * Extract_Usid( char * &txt, int len )
{
    static char buf[16];
    int         ii,  ll;
    char    * uid,   ch;

    buf[0] = 0;
    if (len > 3 && txt[0]=='*' && txt[1]=='[') {        // When a user Ident is included in the text.
        ii = 2; ll = 0; ch = txt[2];
        while (ii<len && ll<15 && ch != ']' && ch ) {   // Loop on the text character until ']'.
            if (ch > ' ') buf[ll++] = ch;               // We skip any space inside the User Identifier.
            ch = txt[++ii];                             // We skip to the next character.
        }
        buf[ll] = 0;                                    // We append the terminal null character.
        if (ch == ']' && ll > 0) {                      // When we have a true user ident ...
            uid = buf;                                  // We return the founded uid to caller.
            ch = txt[++ii];                             // we get the following character.
            if (ch == '.') ch = txt[ii++];              // We skip a period ...
            while (ch == ' ') ch = txt[ii++];           // ... and any successive space.
            if (ch) txt += ii;                          // When have no other character ...
               else txt = NULL;                         // We conclude for an empty slide's text.
        } else  uid = NULL;                             // If the sequence was not a legal uid ...
                                                        // ... we take amll the sequence as slide's text.'
    }
    return uid;
} // static void Extract_Usid( char * ctx, char * &txt, char * &uid ).



static const char * ExtractUsid( char * uid, int nf = 0, int ns = 0 )
// Routine to extract Usid from old Slide Text.
{
    static char buf[16] = "",
                frm[16] = "";
    static int iv,  lv;

    int     bs, ii, lp;
    char      ch, * pt;

    buf[0] = 0;
    if (uid && uid[0]) {
        bs = lp = lv = iv = 0;
        ii = 0;
        while (ch = uid[ii++]) {
            if (bs && ch >= '0' && ch <= '9') break;
            if (ch = '/') bs = 1;
        }
        if (ch) {
            lp = ii - 1;
            do {
                iv *= 10; iv += ch - '0';
                lv++;
                ch = uid[ii++];
            } while (ch && ch >= '0' && ch <= '9');
        }
        if (ch) pt = uid - 1; else pt = NULL;

        if (lv) {
            if (pt)
                snprintf( frm, 15, "%%%*s%%%dd%s", uid, lv+1, pt );
            else
                snprintf( frm, 15, "%%%*s%%%dd", uid, lv+1 );
        } else { strncpy( frm, uid, 15 ); frm[15] = 0; }

    } else if (frm) {
        if (lv) snprintf( buf, 15, frm, ++iv );
           else {
               if (nf && ns) snprintf( buf, 15, "%s:%2d/%2d", frm, nf, ns );
                        else strncpy( buf, frm, 15 );
        }
    }
    if (buf[0]) return buf;
           else return NULL;
} // static char * ExtractUsid( char * &ctx ).



static void ErrorCatch( int ierr )
{
    siglongjmp( JmpBuf, ierr );
} // void ErrorCatch( int ierr ).



int  DirEntry::ReadCntxFile( const char * path )
{
    SliEntry                     * sli;
    int  idf, ii = 0, nsli, ncv,  flgs;
    char      * fnam,   * fnm,   * txt;
    const char                   * uid;
    int icmp, vofl, rdep, cdep, prm[6];
    int                             ln;

    txt = NULL;
    CntxFile.InsErrCatcher( ErrorCatch );
    idf = CntxFile.Open( path, 0 );     // Try to open a related context file.
    if (idf > 0) {                      // Open is a Success (format number idf.
//printf( " ReadCntx : file opened \"%s\"\n\n", path );  fflush( stdout );
        if (!sigsetjmp( JmpBuf, 1 )) {  // While no IO error.
//printf( "A md %d, %d\n", idf, cdeep );  fflush( stdout );
            cdep = LDeep();             // Get the deep of our directory descriptor.
            switch (idf) {
                case 1:  // Version 1.001  Release A01
                    rdep = CntxFile.GetInt();// Get the deep of directory descriptor.
                    vofl = (cdep == 0);
                break;
                // Version 1.000
                case 4:  // Version 1.000  Release A00
                case 3:  // Version 1.000  Release A01 & A02
                case 2:
                    if (idf<5) vofl = CntxFile.GetInt();// Get the volume flag mode.
                               else vofl = (cdep)? 0: 1;
                // We cannot dectect the overlayed volume for old context file versions.
                break;
                default: ;
            }
//printf( "Abis mode %d\n", vofl );  fflush( stdout );
            if (vofl) {                                 // Specific to volume root directory.
                // Get and save for VolEntry if we are in Volume root directory.
                VolFlags = CntxFile.GetInt()&~Context_Update;   // Get the volume flags and text.
                txt  = CntxFile.GetText(TxStr);         // Get the volume name string.
                VolVname = (txt&&txt[0])? strdup( txt ) : NULL;
//printf( " idf = %d\n", idf );
                if (idf == 1) {                         // For complete volume mode (SubVersion > 0).
                    VolPrgPc = CntxFile.GetInt();       // Get Initial position open/select volume.
                    VolPrgId = CntxFile.GetInt();
//printf( " Cntx read RdC Flg %d, Pc %d, Id %d\n", VolFlags, VolPrgPc, VolPrgId );
                }
            }
//printf( "C\n" );  fflush( stdout );
            // Read Directory context setup.
            FlgVal( CntxFile.GetInt()&~Context_Update );// Set directory flags.
            if (idf > 1) rdep=CntxFile.GetInt();        // Set the read deep.
            txt = CntxFile.GetText( TxStr );            // Get and set the directory related text.
            SText( (txt)? txt : NULL );
//printf( "D\n" );  fflush( stdout );
            nsli = NSli();
            // Scan for each slide of this directory.
            if (nsli = NSli()) {                        // When we have some slide(s) in this directory.
                do { // We read a slide record.
                    fnm = CntxFile.GetText( FnStr );    // Get the slide filename.
                    ncv =  0;
                    for(int j = 0; j< 6; j++) { // Loop to get orientation, slide #, film #, ...
                        prm[j] = CntxFile.GetInt();
                        if ((j > 1)&&(prm[j])) ncv++;   // ... Chapter #, and volume #.
                    }
                    txt = CntxFile.GetText( TxStr );    // Get the slide anotation text.
                    // We look for a slide that matchs.
                    do {
                        sli = Slide( ++ii );            // Get the current slide descritor ...
                        fnam = sli->FName();            // ... and its filename.
                        icmp = fl_utf_strcasecmp( fnam, fnm );
                    } while ((ii < nsli)&&(icmp < 0));
                    if (icmp == 0) {            // When filenames match ...
                        sli->FlgVal( prm[0]&~Context_Update );  // ... set the slide flags, ...
                        sli->OriOpe( prm[1] );  // ... the Slide original orientation and the ...
                        sli->CurOpe( prm[1] );  // ... ini        tial orientation (Image_REF part).
                        sli->Numb( prm[2] );    // Set the slide #.
                        sli->Film( prm[3] );    // Set the slide film #.
                        sli->Chap( prm[4] );    // Set the Slide chapter #.
                        sli->Note( prm[5] );    // Set the slide Quality #.
                        sli->Text( (txt)? strdup( txt ):NULL );     // Get the related slide text when present.
                        // Set the flags for present context informations.
                        if (prm[1]) sli->FlgSet( Context_Orient );
                        if (ncv) sli->FlgSet( Context_Slinbr );
                        if (txt) sli->FlgSet( Context_SText );
                    }
                } while (ii < nsli);            // Stop loop on EOF (by use of longjmp) or ...
                                                // ... on end oh SliEntry list.
            }
        }
//printf( "Cntx Close\n" );  fflush( stdout );
        CntxFile.Close();
    } else return 0;

    CntxFile.InsErrCatcher();

    // We return 0 when the deep are egal (No error), ...
    // ... a negative value when the current volume include a previous one ...
    // ... and a positive value when the current volume is include in a previous one.
    // The overlay error must be avoid to preserve the context data.
    return rdep - cdep;
} // int ReadCntxFile( const char * fname, int rfl ).



void DirEntry::WriteCntxFile( const char * path )
{
    SliEntry     * sli;
    int deep = LDeep();

//printf( " WriteCntx : deep = %d, path = \"%s\"\n", deep, path );
    if (!FlgTst( Context_Update|Context_SubDir|Context_Data )) return;
    CntxFile.InsErrCatcher( ErrorCatch );
    if (CntxFile.Open( path, 1 ) > 0) {              // Open to write Success.
        if (!sigsetjmp( JmpBuf, 1 )) {
            CntxFile.PutInt( deep, 2 );              // Write the current directory deep.
            CntxFile.PutEoln();
            if (deep == 0) {
                // Put for VolEntry if we are in Volume root directory.
//printf(" WrtCntx : A VName = \"%s\"\n", (VolVname)? VolVname : "<>" ); fflush( stdout );
                CntxFile.PutInt( VolFlags, 6 );      // Put Volume flags.
                CntxFile.PutText( VolVname );        // Put the Volume name (do not delete!).
//printf(" WrtCntx : B VName = \"%s\"\n", (VolVname)? VolVname : "<>" ); fflush( stdout );
                VolVname = NULL;
                CntxFile.PutInt( VolPrgPc, 6 );      // Write the last position pc and index.
                CntxFile.PutInt( VolPrgId, 6 );
                CntxFile.PutEoln();
            }
            // Put the directory flags and attached text.
            CntxFile.PutInt( FlgVal() & ~Context_Update, 6 );
            CntxFile.PutText( SText() );
            // Create each slide context when required.
            for(int i = 1; i <= NSli(); i++) {
                sli = Slide( i );
                if (sli->FlgTst( Context_Data )) { // Some data must be written (or rewritten).
                    CntxFile.PutText( sli->FName() );
                    CntxFile.PutInt( sli->FlgVal() & ~Context_Update, 6 );
                    CntxFile.PutInt( sli->CurOpe(), 2  );
                    CntxFile.PutInt( sli->Numb(), 6 ); CntxFile.PutInt( sli->Film(), 6 );
                    CntxFile.PutInt( sli->Chap(), 4 ); CntxFile.PutInt( sli->Note(), 2 );
                    CntxFile.PutText( sli->Text() );
                }
            }
        }
        CntxFile.Close();
    }
    CntxFile.InsErrCatcher();
} // int WriteCntxFile( const char * path ).



void DirEntry::WriteDirCntx()
{
    int nd = NDir();
///     ns = NSli();
    DirEntry  * dir;

    if  (FlgTst( Context_Update|Context_SubDir )) {
        // If somethings changed in the slide volume. 
        if (nd) { // Loop for recursive call on all slide sub-directories.
            for(int i=1;i<=nd;i++) {
                dir = SbDir( i );                       // For each sub-directory.
                if (dir->FlgTst( Context_Update|Context_SubDir )) {
                    // If the directory (or su-directories) context was updated.
                    CPath->Set( dir->FName(), 1 );      // Set the Directory path.
                    dir->WriteDirCntx();                // Create the local context file.
                    CPath->Rem();                       // Restore previous context.
                }
            }
        }
        if (/*ns&&*/FlgTst( Context_Update|Context_SubDir )) {
            CPath->Set( CNTX_FNAME );
            WriteCntxFile( CPath->Path() );        // Write context for the current directory.
            CPath->Rem();
        }
    }
} // int DirEntry::WriteDirCntx().



//  Slide reference list Creator (to create a list of slide in a new ordering.
DirRefer::DirRefer( DirRefer * old )
{
    rfd_own_     =         old; // Save the top of directory reference stack, ...
    rfd_frs_ = rfd_lst_ = NULL; // ... initialize the slide reference list header ...
    rfd_nsl_     =           0; // ... and its size (in slide reference).
    RefStack     =        this;
} // DirRefer::DirRefer().



// Suppress a slide reference in the slide reference list.
DirRefer * DirRefer::DelRef( DirRefer * rdi )
{
    SliRefer * rsl;

    if (rdi) {
        RefStack = rdi->rfd_own_;
        while (rfd_frs_) {
            rsl       =       rfd_frs_; // Get the current slide reference.
            rfd_frs_  =  rsl->rfs_nxt_; // keep its next one.
            rsl->rfs_sli_   =     NULL;
            delete[] rsl->rfs_nam_;
            delete rsl;
        }
        delete rdi;
    }  else RefStack = NULL;
    return RefStack;
} // DirRefer * DirRefer::DelRef( DirRefer * rdi ).



// Append a new slide reference in a slide reference list.
void DirRefer::AddSliRef( const char * name, SliEntry * sli )
{
    SliRefer * rsl = new SliRefer;

    rsl->rfs_nxt_ = NULL;
    if (rfd_frs_) rfd_lst_ = (rfd_lst_->rfs_nxt_ = rsl);
    else rfd_frs_ = rfd_lst_ = rsl;
    rfd_nsl_++;
    rsl->rfs_nam_ = strdup( name );
    rsl->rfs_sli_ = sli;
} // void DirRefer::AddSliRef( const char * name, SliEntry * sli ).



void DirRefer::WriteDirCntxFile( int deep, const char * fpath )
// Special procedure to create the directories ranking context ...
// ... of the new sorted slide collecetion (or volume).
//
{
    DirRefer::SliRefer * rsl;
    SliEntry           * sli;

    CntxFile.InsErrCatcher( ErrorCatch );
    if (CntxFile.Open( fpath, 1 ) > 0) { // Open to write Success.
        if (!sigsetjmp( JmpBuf, 1 )) {
            if (!rfd_own_) { // We create the context file of the new root volume.
                CntxFile.PutInt( 0, 2 );        // Deep of 0 for the volume.
                CntxFile.PutEoln();
                CntxFile.PutInt( Context_RkgEna, 6 );     // Volume flags.
                CntxFile.PutText( tvol_nam );   // Volume VName.
                CntxFile.PutInt( 0, 6 );        // Initial value for Volume PrgPc and
                CntxFile.PutInt( 0, 6 );
                /* /// ???
                 *                CntxFile.PutText( dirfrm ); CntxPutText( fold_hde );
                 *                CntxFile.PutText( slifrm );
                 */
            } else CntxFile.PutInt( deep, 2 );  // For none root directory put the dir deep.
            CntxFile.PutEoln();
            // Create the current directory context data.
            CntxFile.PutInt( 0, 6 );            // Put Directory flags.
            ///         CntxPutInt( deep, 4 );
            CntxFile.PutText( NULL );           // Put Directory text.
            // Create the context data for each slide in this new slides directory.
            rsl = rfd_frs_;
            while (rsl) {
                CntxFile.PutText( rsl->rfs_nam_ );
                sli = rsl->rfs_sli_;
                CntxFile.PutInt( sli->FlgVal() & ~Context_Update, 6 );
                CntxFile.PutInt( sli->CurOpe(), 2  );
                CntxFile.PutInt( sli->Numb(), 6 ); CntxFile.PutInt( sli->Film(), 6 );
                CntxFile.PutInt( sli->Chap(), 4 ); CntxFile.PutInt( sli->Note(), 4 );
                CntxFile.PutText( sli->Text() );
                rsl = rsl->rfs_nxt_;
            }
        }
        CntxFile.Close();
    }
    CntxFile.InsErrCatcher();
} // void DirRefer::WriteDirCntxFile( int deep ).



void DirEntry::RkgClear()
// Clear classement data of all slides of a current directory.
{
    int ns = NSli();

    for (int i = 1; i <= ns; i++) Slide( i )->RkgClear();
    FlgClr( Context_Slinbr );
    FlgSet( Context_Update );
} // DirEntry::RkgClear().


/*
void SliEntry::InitBltPath( VolEntry * vol )
// Init the Reverse Path system at the Root ofslide volume.
//
{   // Create and Initialize the path slide access to the volume directory.
    CPath = new FilePath( vol->TDeep() + 1 );
//  CPath->Set( rootd_->FName(), 1 );   // ... Set the volume root directory path.
} // void VolEntry::InitBltPath().



void SliEntry::ReleaseBltPath()
{
    delete CPath;
    CPath = NULL;
} // VolEntry::ReleaseBltPath().
*/


static VolEntry * rvol;

void DirEntry::BltPath()
// Sub-Routine of the function SliEntry::BltPAth().
{
    if (deep_) {
        owner_->BltPath();      // Recursive call to find the volume root.
        CPath->Set( FName(), 1 );
    } else {
        CPath->Clear();         // For Root, erase and put root filename to begin the Path.
        CPath->Set( FName(), 1 );
        rvol = (VolEntry *)owner_;
    }
} // void DirEntry::BltPath().



const char * SliEntry::BltPath( UInt16 * vid )
// Routine to call to get the filepath of the current slide ...
// ... without follow the way from the root directory of the volume ...
// ... allow a random access at any slide of the source volume.
{
    owner_->BltPath();          // Call to build directories path make, ...
    CPath->Set( FName(), 0 );   // ... append the slide filename and ...
    if (vid) *vid = rvol->volrf_->iide;
    return  CPath->Path();      // Return the complete slide file path.
} // char * SliEntry::BltPath().




//
// Functions for VolEntry class.
//


int  VolEntry::SetVolIIde()
{
    if ((IIdeUniqueCnt&IIdeVolumMask) < IIdeVolumMask) {
        if (IIdeVolClear) IIdeUniqueCnt &= IIdeVolumMask;
        IIdeUniqueCnt +=  IIdeVolumIncr;
        ident_ = IIdeUniqueCnt&IIdeVolumMask;
//printf( " Volume %10x name <%s>\n", ident_, fname_ );
        return 1;
    } else {
        IIdeOverflow    |=   4; // Flag a Volume identifier overflow.
        return 0;
    }
} // VolEntry::SetVolIIde().



VolEntry::VolEntry( const char *name, DirEntry * root , VolRef * rf )
{
    volrf_     =    rf;
    if (rf) rf->volm = this;            // Set the reverse volume pointer.
    rootd_     =  root;                 // Set the link to the root directory.
    root->Owner( (DirEntry*)this );     // Set the back link to the Volume record.
    scprg_     =  NULL;
    ident_     =     0;
    tdeep_     =     0;
    nwind_     =     0;
    tndir_     =     0;
    tnsli_     =     0;
    index_     =     0;
    flags_     =     0;
    prcnt_     =     0;
    pridx_     =     0;

    OpenVolTable.AddObj( (void*)this ); // Append the new volume to the open volume table.

} // VolEntry::VolEntry( const char *name, DirEntry * root = NULL, int idx = 0 ).



VolEntry::~VolEntry()
{
    OpenVolTable.RemObj( (void*)this ); // Remove the volume of the open volume table.
    if (volrf_) volrf_->volm = NULL;    // Now remive the reverse VolRef link.
    if (rootd_) delete rootd_;          // Free all the volume tree.
//  if (stext_) delete[] stext_;        // No specific text for volume (use Dir. text).
} // VolEntry::~VolEntry().



void VolEntry::VName( const char * nw_name )
{
    char * name = volrf_->name;
    if (name) delete[] name;
    if (nw_name&&nw_name[0]) name = strdup( nw_name );
                        else name = NULL;
    volrf_->name = name;
    FlgSet( Context_Update );
} // void VolEntry::VName( const char * name ).



void VolEntry::SetVolContext()
//
// To use when the volume structure is build after the root directory.
//
{
    FlgVal( VolFlags );
//printf( "SetVolCntx :  Set VName at \"%s\" for \"%s\" =>\n", VolVname, rootd_->FName() );
    if (volrf_ && !(volrf_->name) && VolVname) VName( VolVname );
    VolVname    = NULL;
    prcnt_  = VolPrgPc;
    pridx_  = VolPrgId;
} // VolEntry::SetVolContext().




void VolEntry::WriteVolContext()
{
//printf( " WVol : A\n" ); fflush( stdout );
    if (FlgTst( Context_Update )) {
        DirEntry * root = VRoot();
        VolFlags =  FlgVal()&Context_Viewer;
        VolVname =  VName();
//printf(" WVol : B VName = \"%s\"\n", (VolVname)? VolVname : "<>" ); fflush( stdout );
        VolPrgPc =  PrgPc();
        VolPrgId =  PrgId();
//printf(" WVol : C VName = \"%s\"\n", (VolVname)? VolVname : "<>" ); fflush( stdout );
        CPath = new FilePath();
        CPath->Set( root->FName(), 1 );
//printf(" WVol : D VName = \"%s\"\n", (VolVname)? VolVname : "<>" ); fflush( stdout );
        root->WriteDirCntx();
//printf(" WVol : E VName = \"%s\"\n", (VolVname)? VolVname : "<>" ); fflush( stdout );
        delete CPath;
//printf(" WVol : F VName = \"%s\"\n", (VolVname)? VolVname : "<>" ); fflush( stdout );
        VolFlags =    0;
        VolVname = NULL;
        VolPrgPc =    0;
        VolPrgId =    0;
    }
} // void VolEntry::WriteVolContexts().



void VolEntry::SetVolIdents()
//
// To set the various identifier in a slide volume.
//
{
    this->SetVolIIde();         // Set a volume identifier.                   
    rootd_->SetDirIdents();     // Set identifiers for each directory and slide of volume.
    rootd_->Owner( (DirEntry*)this );
} // VolEntry::SetVolIdents().






//
// end of <PW-Id>.
//
