//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// DiaViewer Ranking module.
//
//

#include <string.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <math.h>

#include <sys/types.h>
#include <sys/stat.h>


#include <FL/Fl.H>
#include <FL/fl_ask.H>
#include <FL/fl_utf8.h>
#include <FL/fl_message.H>
#include <FL/Fl_File_Chooser.H>
#include <FL/Fl_Native_File_Chooser.H>
#include <FL/Fl_Text_Buffer.H>


#include "SetUpMan.h"

#include "DiaViewer_DIA.h"
#include "DiaViewer_RkgUI.h"

#include "DiaViewer_RKG.h"
#include "DiaViewer_RCT.h"
#include "DiaViewer_SRV.h"
#include "Text_Display.h"



// Define the film structure type sa the base element of theClassification
//
//
typedef struct {                // Define the film structure.
    UInt16    flnb, size, flag; // film#, Size and flag related to a film
    SliEntry           ** slst; // The array of film slides.
} FlmEntry;




// #define GL_DEBUG

// Define the default value of the string in Slide and Folder formats.
char                  Null    =      0;  
char           IniSliHde[]    =   "Ph",
               IniSliSep[]    =    "-",
               IniDirHde[]    =  "Gph",
               IniTrgFnm[]="my_slides",
               IniDirTra[]    = {Null};

int             slid_dig      =      3, // Number of figures for Slide#,
                film_dig      =      3, // Number of figures for Film#,
                fold_dig      =      2, // Number of figures for Directory/folder,
                flmn_max      =     64, // Maximum value of film number (#),
                slin_max      =     64, // Maximum value of slide number (#),
                dirslisz      =     50, // Max. number of slides/folder in the target volume,
                dirdirsz      =     10; // Max. number of Folder/folder in the target volume.

char          * slid_hde      =   NULL, // Slide filename header string,
              * slid_sep      =   NULL, // Slide filename separator string,
              * fold_hde      =   NULL, // Folder/Directory filename header string.
              * fold_tra      =   NULL, // Folder/Directory filename trailer string.
              * tvol_pth      =   NULL; // Initial file path of target volume.

static char   * slid_tra      =   NULL; // Slide trailer (internal use only.)


static int      RankInit      =      1, // To perform the init sequence at the first call.
                Rank_OK       =      0, // Flag to allow the ranking process.
                wind_id       =      0, // Base window index.
            UniquRankAccess   =      0; // Static value to lock or unlock the ...
                                        // ... classement/Ranking functions access.
static VolEntry   * SrcVol    =   NULL; // The Source volume.

static DirEntry   * SrcCDir   =   NULL; // Source current directory.

static SliEntry  ** SliTable  =   NULL, // The table of all slides of the volume.
                *** FlmTable  =   NULL; // The table of all film tables of the volume.
static UInt16     * FlmIdeNb  =   NULL, // The table film #.
                  * FlmRawNb  =   NULL, // Film # for each map raw.
                  * FlmRawSz  =   NULL; // Film # size for each film.

static FilePath   * DstPath   =   NULL; // Path to create the Ranked destination volume.

static int        * DeepTab   =   NULL; // The Deep table to manage the Destination Directories.

static int          DstDeep   =      0, // The destination sub-directory deep.
                    NumbrSli  =      0; // Total number of slide of the source volume.

static char           RootPathStr[256]; // The Target Volume root path is keep here.

static int                      FlmIdx, // Use as index of end of target root dir path (without filename).
                          StartTabLine, // Index of the first line of ranking slides map.
             Frs_Flm, Flm_Siz, Flm_Esz, // First film # and total  and effective number of films, ...
               Sli_Siz,       Sli_TbSz; // ... maximum slide number and total table size.

static DirRefer      * CurDref  = NULL; // Current pointer reference directory and slides.



//
// Define the format to use to create slides and directories names.
//

const char slifrm[] ="%s%0*d%s%0*d%s%s",// Slide file name format.
           dirfrm[] =   "%s_%d_%0*d%s"; // Directory filename format.


// Define MkDir as the same for all operating system.
#if defined( WIN32 ) && !defined( _CYGWIN_ )
# define MkDir( p ) mkdir( p )
#else
# define MkDir( p ) mkdir( p, S_IRWXU|S_IRGRP|S_IROTH )
#endif

#define MAX_SLI_DISPLAY 256


/*
// Must be global for Text_DisplayUI module.
void TxtSave( Fl_Text_Buffer * tbf )
{
    char * fpath = fl_file_chooser( "Save File as", "*.txt", "DiaViewer_Rank_Check.txt", 0 );
    FILE *   fp;
    char * text = tbf->text();
    char     ch;
    int   i = 0;

    text = tbf->text();
    if ((fp = fl_fopen( fpath, "w" ))&&text&&text[0]) {
        while (ch=text[i++]) {
            putc( ch, fp );
        }
        fclose( fp );
        delete text;
    } else {
        fl_alert( "Error : Cannot create the file %s\n\t error message %s\n",
                  fpath, strerror( errno ) );
    }
    
} // void TxtSave().
*/


static void RequiredDirEval( int nsli, int &deep, int &dirnb )
//
// Routine to evaluate the suited values to build the dirname format.
//
{
    int                    ncnt,  ndir;

    // Compute the required sub-directory deep level.
    ndir = (nsli + dirslisz - 1)/ dirslisz;     // Get the maximum number of final directories.
    if (ndir > 1) {                             // When the root directory is not enough, ...
        ncnt = dirdirsz;
        deep = 1;                               // ... one deep allow <dirdirsz> sub-directories.
        while (ncnt < ndir) {                   // Loop to determine the Total sub-directory ...
            ncnt *= dirdirsz;  deep++;          // ... that is required.
        }
        // Here, deep is the required sub-directory deep level.
        // Now, we must determine the suited number of figures ...
        // ... to put the ordering number of each directory.
        ncnt = 1; dirnb = 0;
        while (ncnt < ndir) { ncnt *= 10; dirnb++; }
    } else { deep = 0; dirnb = 0; }            // We use only the root directory.

} // static void RequiredDirEval( int &tdep, int &ndf ).



static int FlmSliCount( DirEntry * dir )
// Recursive function to compute the total number of ...
// ... numbered slides without the Lock Ranking flag set.
//
{
    int    nd   =  dir->NDir(),
           ns   =  dir->NSli(),
           efs  =            0,
                flid,     slid;
    SliEntry             * sli;

    for (int i=1; i<=nd; i++)
        efs += FlmSliCount( dir->SbDir( i ) );
    for (int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        flid = sli->Film(); slid = sli->Numb();
        if (flid&&slid&&(!sli->FlgTst( Context_LckRank ))) {
            // Set Frs_Flm as the first ranking enable film number, and ...
            // ... Flm_Siz as the related last film number. Set also, ...
            // ... Sli_Siz as the maximum slides index in any ranking valid film ...
            // ... or partial film (in an incremental mode). The previously loaded slides ...
            // ... in the new volume are indicated by the Context_LckRank flag. 
            if (!Frs_Flm) Frs_Flm = flid;
                     else if (Frs_Flm > flid) Frs_Flm = flid;
            if (Flm_Siz < flid) Flm_Siz = flid;
            if (Sli_Siz < slid) Sli_Siz = slid;
            efs++;
        }
    }
    return efs;
} //static int FlmSliCount( DirEntry * dir ).



static void RequiredSliEval( int nsli, int &flmdg, int &slidg )
//
// Routine to evaluate the suited values to build the slide and ...
// ... directories filename format.
//
{
    int     qflm,  qsli,  qusl,
            nfld,  nsld,  esli,
                           nqu;

/*
    nfld = flmdg; qflm = 10;    // Compute the total capacity for the number of films.
    while (--nfld) qflm *= 10;
    qflm--;                     // A zero film number is not allowed.
    if

    nsld = slidg; qsli = 10;    // Compute the total capacity for the number of slides/film.
    while (--nsld) qsli *= 10;
    qsli--;                     // A zero slide number is not allowed.

*/
    static int         res = 0;

    if (res) return;

    nfld = flmdg; qflm = 10;    // Compute the total capacity for the number of films.
    while (--nfld) qflm *= 10;
    qflm--;                     // A zero film number is not allowed.

    nsld = slidg; qsli = 10;    // Compute the total capacity for the number of slides/film.
    while (--nsld) qsli *= 10;
    qsli--;                     // A zero slide number is not allowed.

    // Compute the effective number of ranked slides in the source volume.
    Frs_Flm = Flm_Siz = Sli_Siz = 0;
    esli = FlmSliCount( SrcCDir );
    Flm_Siz = Flm_Siz - Frs_Flm + 1; // Now Flm_Siz is the number of film used for ranking.

    // When the total number of slide in the source volume is greater than ...
    // ... the global capacity of films and slides/film, it is better to ...
    // ... change the figures numbers flmdg and/or slidg.
    // In this case, we suggest to increase the films number capacity.
    // When the number of effective numbered of slides is less than this capacity, ...
    // ... we modify the film capacity and stop the ranking process for agreement.
    //
    nqu = 10;
    qusl = qsli*qflm;
    if (qusl < nsli) {  // The posiible slide number is too large =>warnig or error.
        qflm = (esli + qsli - 1)/qsli;          // Now qflm is the minimum capacity and we ...
        res = 1;
        while (nqu <= qflm) { nqu*=10;res++; }  // ... return a new value (to flag warning).
    } else res = 0;

    // Send diagnostic to user.
    if (res) {
        if (qusl < esli) {
            fl_alert(   "Error :\n\tFor a total of %d slides with %d numbered slides,\n\t%s\n\t%s%d%s",
                        nsli, esli,
                        "you must change the film or slide figures number(s) as (for example) :",
                        "set figures film number at ", res, " to keep all source volume slides." );
            Rank_OK = 0;
        } else {
            fl_message( "Warning :\n\tFor a total number of %d slides, you should change\n\t%s%d.\n\t%s",
                        nsli, "(for example) the film figures number at ", res,
                        "The actual setting do not authorize to keep all possible source volume slide." );
            Rank_OK = 1;
        }
    }
    res = 0;
} //static void RequiredSliEval( int nsli, int &flmdg, int &slidg ).



static void EditSliExa()
{
    char str[32];

    snprintf( str, 31, slifrm, slid_hde, film_dig, 5, slid_sep, slid_dig, 3, slid_tra, ".jpeg" );
    cs_example->value( str );
} //static void EditSlirExa().




static void EditDirExa()
{
    char str[32];

    snprintf( str, 31, dirfrm, fold_hde, DstDeep, fold_dig, 9, fold_tra );
    cf_example->value( str );
} //static void EditDirExa().



static char * NumWidth( int nbd )
{
    static char str[64];
    int           i = 0;

    while (i < nbd) { str[i] = '0'+ i; i++; }
    str[i] = 0;
    return str;
} // static char * NumWidth( int nbd ).



void Rkg_Setup( DiaViewer_GL * wgl )
// Call to enter in the Classification sub-system.
//
{
    char       str[64];
    char          * s1;
    int   len,     did,
          tdp,    nsli;

    if (UniquRankAccess) {                      // Classement is already in use.
        fl_alert( "You cannot simultaneously activate the ranking subsystem." );
    } else {
        SrcVol = wgl->Curr_Vol;                 // Keep the Source volume.
        if (!SrcVol) return;                    // A current volume must be opened.

        UniquRankAccess     =    1;             // Lock the access to ranking funtion use.
        wind_id = wgl->ViewIndex() + 1;         // Get the base View index.

        OpenRkgSetup( svargc, svargv,wind_id ); // Create the specific ranking window.

        if (RankInit) {                         // To perform an init sequence ...
            Rank_OK  = 1;                       // Assume success.
            RankInit = 0;                       // ... only for the first call.
            slid_tra = strdup( "-m" );          // Define the middle index trailer for normal slides.
        }
        if (!slid_hde) slid_hde = IniSliHde;
        if (!slid_sep) slid_sep = IniSliSep;
        if (!fold_hde) fold_hde = IniDirHde;
        if (!fold_tra) fold_tra = IniDirTra;

        // Define the source volume access.
        SrcCDir =  SrcVol->VRoot();             // Get the source root directory.
        rkg_volsrc->value( SrcCDir->FName() );  // Display the name/path of source volume.

        // Define the default target volume root.
        if (tvol_pth&&tvol_pth[0]) {            // Use the previously use dir. as default ...
            strncpy( RootPathStr,tvol_pth,248 );// ... when defined ...
            len = strlen( RootPathStr );
            if (RootPathStr[len-1] != DIRSEP) RootPathStr[len++]=DIRSEP;
        } else {                                // ... else use the user's home directory.
            if (tvol_pth) { delete[] tvol_pth; tvol_pth = NULL; }
            SetUpSolveLogPath( UserDir, s1, len );
            strncpy( RootPathStr, s1, 248 );
            s1 = RootPathStr;                   // Append a directory separator when not present.
            if (s1[len-1] != DIRSEP) s1[len++]=DIRSEP;
            s1[len] = 0;
            FlmIdx = len;                       // Keep the size of path (without dir. name).
            strncpy( s1+len,IniTrgFnm,255-len );// Append the default root filename.
        }
        rkg_root->value( RootPathStr );         // Set it, in the setup window.

        NumbrSli = nsli = SrcCDir->CnSli();     // Get the total possible count of slides.

        RequiredDirEval( nsli, tdp, did );      // Evaluate the minimum deep and fold_dig.

        DstDeep  = tdp;
        if (did > fold_dig) fold_dig = did;     // We set the minimum value if too small.
        cf_nbfd->value( fold_dig );             // Display the value (changed if too small).
        cf_nbfd->minimum( did );                // Set the minimum required.
        cf_head->value( fold_hde );             //
        cf_trailer->value( fold_tra );          //
        cf_dirnb->value( NumWidth( fold_dig ) );
        EditDirExa();                           // Display the folder name example.

        cs_mxflm->value( flmn_max );
        cs_mxsli->value( slin_max );
        sprintf( str, "corresponding to\n %d slides max.", (flmn_max - 1)*(slin_max - 1) );
        cs_slitmx->value( strdup( str ) );
        cs_head->value(  slid_hde );
        cs_nbflm->value( film_dig );
        cs_flmnb->value( NumWidth( film_dig ) );
        cs_separ->value( slid_sep );
        cs_nbsli->value( slid_dig );
        cs_slinb->value( NumWidth( slid_dig ) );
        cs_trailer->value( slid_tra );
        EditSliExa();                           // Display the slide filename example.

        RequiredSliEval( nsli, film_dig, slid_dig ); // Check for Rank_Ok.

    }
} // Rkg_Setup( int argc, char** argv ).



void Rkg_Maximas( int fl, int val )
{
    int tmsli;
    char str[64];

    if (fl) if (val*slin_max > IIdeMaxSli) cs_mxflm->value( flmn_max );
            else flmn_max = val;
       else if (val*flmn_max > IIdeMaxSli) cs_mxsli->value( slin_max );
            else slin_max = val;       
    tmsli = (flmn_max - 1)*(slin_max - 1);
    snprintf( str, 63, "corresponding to\n %d slides max.", tmsli );
    cs_slitmx->value( strdup( str ) );
} // void Rkg_Maximas( int fl ).



void Rkg_Quit()
// To Quit the classification sub-system.
//
{
/// int len;

    strncpy( RootPathStr, rkg_root->value(), 255 ); RootPathStr[255] = '\0';
/// len = strlen( RootPathStr );
/// while (RootPathStr[len] != DIRSEP) len--;
/// RootPathStr[len] = 0;
    delete rkg_win;
    
    if (tvol_pth) delete[] tvol_pth;
    tvol_pth = strdup( RootPathStr );
    UniquRankAccess   =  0;                     // unlock the classement access.
} // void Rkg_Quit().



//
// Routines/functions to perform the checking of Rank data.
//

/*
static int CountFlmSli( DirEntry * dir )
//
// Recursive routine to size a map sizes of defined maximum film and slide numbers.
//
{
    int nd = dir->NDir(),
        ns = dir->NSli();
    int  cnf, cns, efsli;
    SliEntry       * sli;

    for(int i=1; i<=nd; i++) efsli += CountFlmSli( dir->SbDir( i ) );

    for(int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        cnf = sli->Film(); cns = sli->Numb();
        if (cnf&&cns&&(!sli->FlgTst(Context_LckRank))) {
            if (Flm_Siz < cnf) Flm_Siz = cnf;
            if (!Frs_Flm) Frs_Flm = cnf;
                  else if (Frs_Flm > cnf) Frs_Flm = cnf;
            if (Sli_Siz < cns) Sli_Siz = cns;
            efsli++;
        }
    }
    return efsli;
    // int CountFlmSli( DirEntry * dir )
} // int CountFlmSli( DirEntry * dir.
*/


static void CountFillFilm( DirEntry * dir )
//
// Recursive routine to set the film index table.
//
{
    int nd = dir->NDir(),
        ns = dir->NSli(),
                     cnf;
    SliEntry       * sli;

    for(int i=1; i<=nd; i++) CountFillFilm( dir->SbDir( i ) );

    for(int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        cnf = sli->Film();
        if (cnf&&sli->Numb()&&(!sli->FlgTst(Context_LckRank))) FlmIdeNb[cnf-1]  = cnf;
    }
} /* static void CountFillFilm(). */



static int AnalyseFlmSli( DirEntry * dir )
//
// Routine to create a Chart/Map of defined films and slides numbers.
//
{
    int nd = dir->NDir(),
        ns = dir->NSli();
    int  cnf,  cns,  ner;
    SliEntry       * sli;

    ner = 0;
    for(int i=1; i<=nd; i++) ner += AnalyseFlmSli( dir->SbDir( i ) );

    for(int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        cnf = sli->Film(); cns = sli->Numb();
        if (!(sli->FlgTst( Context_LckRank ))&&
            cnf&&(cnf <= Flm_Siz)&&(cns <= Sli_Siz))
        {
            if (FlmTable[cnf-1][cns-1]) {
                FlmTable[cnf-1][cns-1] = (SliEntry*)8;
                ner++;
            } else FlmTable[cnf-1][cns-1] = sli;
        }
    }
    return ner;
} // static void AnalyseFlmSli( DirEntry * dir ).



static void MouseCallBack( int bt, int raw, int col )
{
    int fraw = raw - StartTabLine,
        fcol = col,
        flmnb = 0, slinb = 0;

    if (fraw >= 0) {
        printf( " Clic with Button %d, at (raw,col) [%d,%d]\n", bt, fraw, fcol );
        fflush( stdout );

        flmnb = FlmRawNb[fraw];
        slinb = (fcol - 12 + 3)/3;
//printf( " ?? Film # = %d, Slide # = %d, film sz = %d\n", flmnb, slinb, FlmRawSz[fraw] );

        if ((flmnb == 0)||(slinb > FlmRawSz[fraw])) flmnb = slinb = 0;
        else {
            printf( " => Film # = %d, Slide # = %d\n", flmnb, slinb );

//          Viewer_Gl[widx]->Search_Init( flmnb, slinb );

        }
        fflush( stdout );
    }
} // static void MouseCallBack( int bt, int row, int col ).



int  Rkg_FlmSliChart( VolEntry * vol, int widx )
// To Create a table of films and slide numbers.
// Routine limited to 64 slide/film(> common length of slide silver-film),
// and 64 films.
//
{
    int   esli,   cflm,    lflm,  nerr,
                    wlrg,  nraw,  ncol;
    SliEntry                      ** p;
    char                   txtbuf[256],
                           txtitle[40];
    Text_Display              * DspTxt;

    if (!vol) return  0;

    int scr_wmax = (screen_w*8)/10, scr_wmin =        550; //screen_w/5
/// int scr_hmax = (screen_h*8)/10, scr_hmin = screen_h/5;

    Frs_Flm = Flm_Siz = Sli_Siz = 0;
    esli = FlmSliCount( vol->VRoot() ); // Get the sizes for table to build.
    Flm_Siz = Flm_Siz - Frs_Flm + 1;    // Now Flm_Siz is the number of film used for ranking.

    // Free all previous Ranking/Sorting slides table.
    if (SliTable) { delete[] SliTable; SliTable = NULL; }
    if (FlmTable) { delete[] FlmTable; FlmTable = NULL; }
    if (FlmIdeNb) { delete[] FlmIdeNb; FlmIdeNb = NULL; }
    if (FlmRawNb) { delete[] FlmRawNb; FlmRawNb = NULL; }
    if (FlmRawSz) { delete[] FlmRawSz; FlmRawSz = NULL; }
    FlmIdeNb = new UInt16[Flm_Siz];
    for (int i = 0; i < Flm_Siz; i++) FlmIdeNb[i] = 0;
    CountFillFilm( vol->VRoot() );
    // Now count of really present films.
    Flm_Esz = 0;
    for (int i=0; i<Flm_Siz; i++) if (FlmIdeNb[i]) Flm_Esz++;

    Sli_TbSz = Flm_Esz*Sli_Siz;
    SliTable = new SliEntry*[Sli_TbSz]; // Allocate  and initialyze the new sorting table.
    for (int i=0;i<Sli_TbSz;i++) SliTable[i] = NULL;
                           
    FlmTable = new SliEntry**[Flm_Siz]; // Allocate the direct films access slide list table ...
    p = SliTable;                       // ... and built it.
    for(int i=0;i<Flm_Siz;i++) {
        if (FlmIdeNb[i]) { FlmTable[i] = p; p += Sli_Siz; }
                    else FlmTable[i] = NULL;
    }
    FlmRawNb = new UInt16[2*Flm_Siz];
    FlmRawSz = new UInt16[2*Flm_Siz];
    for(int r=0; r<2*Flm_Siz; r++) { FlmRawNb[r] = 0; FlmRawSz[r] = 0; }

    nerr = AnalyseFlmSli( vol->VRoot() );

    // New section to support new version of Text_Display module.

    snprintf( txtitle, 38, "DiaViewer Ranking Map from window %d", widx );
    DspTxt = Text_Display::Init( 800, 500, txtitle );

    DspTxt->SetTitleFont( FL_HELVETICA, 14 );
    snprintf( txtbuf, 255, "Ranking volume table %s for window # %d", vol->VRoot()->FName(), widx );
    DspTxt->SetTitle( strdup( txtbuf ) );
    DspTxt->SetCallBack( (CallBack_t)MouseCallBack );
    DspTxt->SetFont( FL_COURIER, 10 );

    wlrg = (int)(18.25*Sli_Siz + 135.25); // Formulae used to determine for courrier font of fixed size 10.
    if (wlrg < scr_wmin) wlrg = scr_wmin;
                    else if (wlrg > scr_wmax) wlrg = scr_wmax;

    printf( " Used Window width = %d\n", wlrg );


    DspTxt->Append( " Source volume %s contains a total of %d slides\n",
                    vol->VRoot()->FName(), vol->TnSli() );
    DspTxt->Append( " and %d slides are numbered to be insert in a new volume to create.\n", esli );
    DspTxt->Append( "\n There are %d films with a maximum of %d slides\n", Flm_Esz, Sli_Siz );
    DspTxt->Append( " The slide above the maximum number of 64 are not take in account in the following map.\n\n" );
    if (nerr) {
        DspTxt->Append( "\t *** Ranking Error : %d %s ***\n\t\t%s\n\t\t%s\n\n", nerr,
                        "\t  *  Some slides share the same numbers (of film and slide).",
                        "\t  *  You must correct this to be allowed to perform the ranking.",
                        "\t  *  These error are flagged in the following map by ** in the films." );

    } else  DspTxt->Append( " You are ready to perform the ranking.\n\n" );
/*
    snprintf( txtbuf, 255,
              "\n\n Source volume %s contains a total of %d slides\n",
              vol->VRoot()->FName(), vol->TnSli() );
    DspTxt->txt_buffer->append( txtbuf );

    snprintf( txtbuf, 255,
              " and %d slides are numbered to be insert in a new volume to create.\n", esli );
    DspTxt->txt_buffer->append( txtbuf );
    snprintf( txtbuf, 255,
              "\n There are %d films with a maximum of %d slides\n", Flm_Esz, Sli_Siz );
    DspTxt->txt_buffer->append( txtbuf );

//  snprintf( txtbuf, 255,
//            " The slide above the maximum number of 64 are not take in account in the following map.\n\n" );
//  DspTxt->txt_buffer->append( txtbuf );

    if (nerr) snprintf( txtbuf, 255,
                        "\t *** Ranking Error : %d %s ***\n\t\t%s\n\t\t%s\n\n", nerr,
                        "\t  *  Some slides share the same numbers (of film and slide).",
                        "\t  *  You must correct this to be allowed to perform the ranking.",
                        "\t  *  These error are flagged in the following map by ** in the films." );
         else snprintf( txtbuf, 255, " You are ready to perform the ranking.\n\n" );
    DspTxt->txt_buffer->append( txtbuf );
*/

    StartTabLine = DspTxt->LineCount();

    lflm = 0; nraw = 0;
    for (int ii=0; ii<Flm_Siz; ii++) {
        p = FlmTable[ii];
        if (p) {
            cflm = FlmIdeNb[ii];
            // Skip one line when the films # are not consecutive.
            if (lflm&&(cflm-lflm > 1)) {
                DspTxt->append( "\n" );
                nraw++;
            }
            lflm = cflm;
            FlmRawNb[nraw] = cflm;
            FlmRawSz[nraw] =    1;
            ncol = 0;
            snprintf( txtbuf, 255, " Film# %3d /", cflm );
            DspTxt->append( txtbuf );
            for (int jj = 0; jj < Sli_Siz; jj++) {
                if (p[jj]) {
                    if (p[jj] != ((SliEntry*)8)) sprintf( txtbuf, " %2d", jj+1 ); 
                                            else if (p[jj]) sprintf( txtbuf, " **" );
                    ncol = jj;
                } else sprintf( txtbuf, " .." );
                DspTxt->append( txtbuf );
            }
            DspTxt->append( "\n" );
            FlmRawSz[nraw] = ncol + 1;
//printf( " nraw = %d, Film # = %d, Film Size = %d\n", nraw, FlmRawNb[nraw], FlmRawSz[nraw] );
            nraw++;
        }
    }
    DspTxt->append( "\n" );

    return (nerr)? 0 : 1;
} // void Rkg_FlmSliChart( VolEntry * vol ).



static void ChangeProtFlg( DirEntry * dir, int flm, int fla, int flv )
{
    int nd = dir->NDir(),
        ns = dir->NSli();
    SliEntry       * sli;

    for(int i=1; i<=nd; i++) ChangeProtFlg( dir->SbDir( i ), flm, fla, flv );

    for(int i=1; i<=ns; i++) {
        sli = dir->Slide( i );
        if (flm == sli->Film()) {
            if (flv) sli->FlgSet( fla );
                else sli->FlgClr( fla );
        }
    }    
} // static void ChangeProtFlg( DirEntry * dir ).



void Rkg_ProtFlg( SliEntry * slb, int pty, int flg )
//
// For all slide of the current film, set(flg=1) or Clear(flg=0) ...
// ... the slide protection(pty=1) or slide ranking lock(pty=0) flags.
//
{
    DirEntry * dir = slb->Owner();
    int        flm =  slb->Film(),
        fla = (pty)? Context_Protect : Context_LckRank;

    while (dir->LDeep()) dir = dir->Owner();     // Locate the volume root directory.
    if (dir)  ChangeProtFlg( dir, flm, fla, flg );
} // void Rkg_PrtFlg( SliEntry * sli, int pty, int flg ).



void Rkg_ProtectMan( SliEntry * sli )
{

} // void Rkg_ProtectMan( SliEntry * sli ).


void Rkg_Verify()
//
// To verify the data coherence for ranking.
//
{
    Rank_OK = (Rkg_FlmSliChart( SrcVol, wind_id ))? 2 : 1;
} // void Rkg_Verify().




//
// Routine/function to perform the ranking.
//

#define BUFSIZE 8192
#define UNIX_IO


static int FileCopy( const char * dstpth, const char * srcpth )
// Copy one file.
// Used to copy the image files from the source volume to the target one.
// It is binary level copy.
{
    int           ierr = 0,
                      size;
    char   buffer[BUFSIZE];

    if (!access( srcpth, F_OK )) {

#ifdef UNIX_IO
        int       src, dst;
        struct stat stasrc;

        if ((src = open( srcpth, O_RDONLY|O_BINARY )) == -1) {
            fl_alert( "Cannot open the slide file \"%s\":\n\t%s", srcpth, strerror( errno ) );
            ierr = -1;            
        } else {
            fstat( src, &stasrc );
            
            if ((dst = open( dstpth, O_WRONLY|O_CREAT|O_TRUNC|O_BINARY, stasrc.st_mode )) == -1) {
                fl_alert( "Cannot create the slide file \"%s\":\n\t%s", dstpth, strerror( errno ) );
                ierr = -1;
            } else {
                while ((size = read( src, buffer, BUFSIZE )) > 0)
                    if (write( dst, buffer, size ) == -1) {
                        fl_alert( "Write Error one slide file \"%s\":\n\t%s", srcpth, strerror( errno ) );
                        ierr = -1; break; 
                    }
                if (size == -1) {
                    fl_alert( "Read Error one slide file \"%s\":\n\t%s", srcpth, strerror( errno ) );
                    ierr = -1;
                }
                close( dst );
            }
            close( src );
        }

#else
        FILE  * src, * dst;

        if (src = fl_fopen( srcpth, "r" )) {
            if (dst = fl_fopen( dstpth, "w" )) {
                while ((size = fread( buffer, sizeof( char ), BUFSIZE, src )) > 0) {
                    if (fwrite( buffer, sizeof( char ), size, dst ) != size) {
                        fl_alert( "IO Error one slide file \"%s\":\n\t%s", srcpth, strerror( errno ) );
                        ierr = -1; break;                        
                    }
                }
                fclose( dst );
            } else {
                fl_alert( "Cannot create the slide file \"%s\":\n\t%s", dstpth, strerror( errno ) );
                ierr = -1;
            }
            fclose( src );
        } else {
            fl_alert( "Cannot open the slide file \"%s\":\n\t%s", srcpth, strerror( errno ) );
            ierr = -1;
        }
#endif
    }
    return ierr;
} // static int FileCopy( const char * dst, const char * src ).



static int TreeCreate( int deep, int idd )
// Recursive Routine to create any path element of a given path ...
// ... and prepare path builting of a new (ranked) volume.
//
{
    int            dii;
    char     dfnam[64];
    const char  * path;

    dii = DeepTab[deep];                // Get the current directory index (or -1 when no existing).
    if (dii != idd) {                   // When the expected directory is not in the current path.
        if (dii >= 0) {                 // When, a previous directory is existing, remove it from path ...
            DstPath->Set( CNTX_FNAME ); // ... after the context file creation.
            CurDref->WriteDirCntxFile( dii, DstPath->Path() );
            CurDref = CurDref->DelRef( CurDref );
            DstPath->Rem();
            DstPath->Rem();
        }
        // Check for parent directory and stop on error.
        if (TreeCreate( deep-1, idd/dirdirsz )) return -1;
        // Build the filename of the expected directory and ...
        snprintf( dfnam, 63, dirfrm, fold_hde, deep, fold_dig, idd, fold_tra );
        DstPath->Set( dfnam, 1 );       // ... put it in the path.
        path = DstPath->Path();         // Keep the complete path and ...
        DeepTab[deep] = idd;            // ... put its index in the deep table.
        if (access( path, F_OK ))       // If the expected directory does nos exist, we create it.
            if (MkDir( path )) return -1;       // On error we return.
//      CurDref = CurDref->NewRef();    // Create the directory/slide reference structure.
        CurDref = new DirRefer(CurDref);// Create the directory/slide reference structure.
    }
    return 0;
} // static int TreeCreate( int deep, int idd ).



static int SlideCopy( SliEntry * sli )
{
    char                    nsname[32];
    const char   * src,  * dst,  * ext;
    int i;

    // Compute the slide index as a combinaison of film and slide numbers.
    int ids = (sli->Film() - 1)*dirslisz + sli->Numb() - 1;
    // Create any subdirectory to be ready to copy the slide.
    if (TreeCreate( DstDeep, ids/dirslisz )) { 
        // On error send the message.
        fl_alert( "Error : Cannot use the path \"%s\"\n\t%s\n", DstPath->Path(), strerror( errno ) );
        return -1;
    }
    src = sli->BltPath();
    i = strlen( src );
    while ((i>0)&&(src[--i] != '.')) ;  // Locate the last dot separator.
    ext = src + i;
    // Build the new filename (in the new ranked volume) of the slide.
    snprintf( nsname, 31, slifrm, slid_hde, film_dig, sli->Film(), slid_sep,
                                            slid_dig, sli->Numb(), slid_tra, ext );
    // Put it in the path and perform the slide copy.
    DstPath->Set( nsname, 0 );
    dst = DstPath->Path();
    if (access( dst, F_OK)) {
        if (FileCopy( dst, src )) {
            // On error send the message.
            fl_alert( "Error : Cannot copy the slide \"%s\"\n\t%s\n", DstPath->Path(), strerror( errno ) );
            return -1;
        }
    }
    DstPath->Rem();
    CurDref->AddSliRef( nsname, sli );
    return 0;
} // static int SlideCopy( const char * path, SliEntry * osli ).




void Rkg_CreateVolume()
//
// Call to define the new ranked volume path.
//
{
    SliEntry             * sli;
    const char          * root;
    int    flcnt, sliid, fdeep;

    if (Rank_OK <= 1) {
        fl_alert( "You cannot perform the Ranking process.\n\t%s",
                  "before to get success in the Data Check test." );
        return;
    }

    root = rkg_root->value();                           // Get the user root path.
    if (tvol_pth) delete[] tvol_pth;
    tvol_pth = strdup( root );
    if (access( root, F_OK|R_OK|W_OK|X_OK )) {
        if (errno != ENOENT) {
            fl_alert( "Error: Can't create a new volume in the path \"%s\"\n\t%s",
                    root, strerror( errno ) );
            return;
        }
///     char question[256];

        if (fl_choice( "The directory %s does not exist.\n\tDo you want create it ?",
                           "Yes and continue", "Quit", 0, root ))
            return;
        else
            if (MkDir( root )) {
                fl_alert( "Cannot create the directory \"%s\"\n\t%s", root, strerror( errno ) );
                return;
            }
    }

    strncpy( RootPathStr, root, 248 );
    DstPath = new FilePath( DstDeep + 1 );              // Create the destination Path and ...
    DstPath->Set( RootPathStr, 1 );                     // ... set it as root of the new volume.

    CPath = new FilePath( SrcVol->TDeep() + 1 );        // Init the Source Filepath.

    DeepTab = new int[DstDeep+1];                       // Create the deep table to manage the ...
    DeepTab[0] = 0;
    for(int i=1; i<=DstDeep; i++)  DeepTab[i] = -1;     // ... volume tree creation.

    CurDref = new DirRefer( NULL );                     // Create the root reference directory.

    flcnt = 0;
    while (flcnt < Flm_Siz) {                           // Loop one all Films.
        if (FlmTable[flcnt]) {                          // When this Film exist.
            sliid = 0;
            while (sliid < Sli_Siz) {                   // Loop on all slide of the same film.
                if (sli = FlmTable[flcnt][sliid])       // Get the SliEntry reference.
                    if (!(sli->FlgTst( Context_LckRank )))
                        SlideCopy( sli );               // Copy all existing slides ...
                                                        // ... in new volume tree.
                sliid++;
            }
        }
        flcnt++;
    }

    // We must close all pending directory.
    fdeep = DstDeep;
    do {
        DstPath->Set( CNTX_FNAME );                     // Create context files for each not closed directories.
        CurDref->WriteDirCntxFile( fdeep, DstPath->Path() );
        CurDref = CurDref->DelRef( CurDref );
        DstPath->Rem();
        DstPath->Rem();
    } while (fdeep-- );

    delete[] DeepTab;
    delete CPath; CPath = NULL;;
    delete DstPath;

    // Now we must open the new slide volume, ...
    // ... set it all its context data and ...
    // ... save them.
    //
    

} // void Rkg_CreateVolume().





//
// Routines to manage the Ranking setup changes.
//

void Rkg_ChangeTrgPath()
{
    strncpy( RootPathStr, rkg_root->value(), 248 );
    if (tvol_pth) delete[] tvol_pth;
    tvol_pth = strdup( RootPathStr );
} // void Rkg_ChangeTargetPath().



//void Rkg_ChangeVolName( const char * vn )
//{
//  SrcVol->VName( strdup( vn ) );
//} // void Rkg_ChangeVolName( const char * vn ).



void Rkg_NVolSelect()
{
    const char * path;
    
    path = DLG_DirChooser( "Folder selection for new vranked volume", RootPathStr );
    if (!path) return;

    if (!access( path, F_OK|R_OK|W_OK|X_OK )) {
        strncpy( RootPathStr, path, 248 );
        rkg_root->value( RootPathStr );
    } else fl_alert( "Cannot get full acces to path \"%s\"\n\t%s", path, strerror( errno ));
} // void Rkg_NVolSelect().



static int CheckChar( char ch )
{
    if (((ch >= 'O')&&(ch <= '9'))||
        ((ch >= 'A')&&(ch <= 'Z'))||
        ((ch >= 'a')&&(ch <= 'z'))) return ch;
    switch (ch) {
        case '/':
        case '\\':
            return DIRSEP;

        case '-':
        case '_':
            return ch;
        default:
            return 0;
    }
} // CheckChar( char ch ).



static char * FiltreChar( const char * str )
{
    static char       buf[256];
    char            ch1,   ch2;
    int       i = 0,     j = 0;

    while ((i < 255)&&(ch1 = str[i])) {
        ch2 = CheckChar( ch1 );
        if (ch2) buf[j++] = ch2;
        i++;
    }
    buf[j] = 0;
    return buf;
} // static char * FiltreChar( const char * str ).



void Rkg_EnterSliHead()
// Call to define the slide head filename (a short string).
//
{
    char * str = FiltreChar( cs_head->value() );

    cs_head->value( str );
    if (slid_hde ) free( slid_hde );
    slid_hde = strdup( str );
    EditSliExa();
} // void Rkg_EnterSliHead().


void Rkg_EnterSliSepa()
// Call to define the separator between film# and slide# (a very short string/char).
//
{
    char * str = FiltreChar( cs_separ->value() );

    cs_separ->value( str );
    if (slid_sep ) free( slid_sep );
    slid_sep = strdup( str );
    EditSliExa();
} // void Rkg_EnterSliSepa().



void Rkg_ChangFlmNbLg()
// Call to define the number of figures for the film#.
//
{
    film_dig = cs_nbflm->value();
    cs_flmnb->value( NumWidth( film_dig ) );
    EditSliExa();
    printf( " FLM\n" );
    RequiredSliEval( NumbrSli, film_dig, slid_dig );
} // void Rkg_ChangFlmNbLg().



void Rkg_ChangSliNbLg()
// Call to define the number of figures for the slide#.
//
{
    slid_dig = cs_nbsli->value();
    cs_slinb->value( NumWidth( slid_dig ) );
    EditSliExa();
    printf( " SLI\n" );
    RequiredSliEval( NumbrSli, film_dig, slid_dig );
} // void Rkg_ChangSliNbLg().



void Rkg_EnterDirHead()
// Call to define the folder head filename (a short string).
//
{
    char * str = FiltreChar( cf_head->value() );

    cf_head->value( str );
    if (fold_hde) free( fold_hde );
    fold_hde = strdup( str );
    EditDirExa();
} // void Rkg_EnterSliHead().



void Rkg_ChangDirNbLg()
// Call to define the number of figures for the slide#.
//
{
    fold_dig = cf_nbfd->value();
    cf_dirnb->value( NumWidth( fold_dig ) );
    EditDirExa();
} // void Rkg_ChangSliNbLg().



void Rkg_EnterDirTrai()
// Call to define the folder head filename (a short string).
//
{
    char * str = FiltreChar( cf_trailer->value() );

    cf_trailer->value( str );
    if (fold_tra ) free( fold_tra );
    fold_tra = strdup( str );
    EditDirExa();
} // void Rkg_EnterSliTrai().






//
// end of <PW-Id>.
//
