//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module to manage the File Locking Input/output Files
//



//
// Common header for all CPU environment.
//

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include <string.h>
#include <stdarg.h>

#include "Service_FLOCK.h"


#if defined( __WIN32 )&&!( __CYGWIN__ )

#  define MTH_FSOPEN
#  include <windows.h>
#  include <unistd.h>
#  include <share.h>

#elif defined(__CYGWIN__)

#  define MTH_FCNTL
#  include <unistd.h>
#  include <fcntl.h>

#elif (defined(__LINUX__)||defined(__LINUX))

// We must choice the best for Network file access ( NFS or SAMBA )
//#  define MTH_FLOCK
#  define MTH_LOCKF

#  ifdef MTH_FLOCK
#    include <sys/file.h>
#  else
#    include <unistd.h>
#  endif

#elif (defined(__UNIX)&&defined(_SGI))

#  define MTH_LOCKF
#  include <unistd.h>

#elif (defined(__UNIX))

// We hope defined MTH_LOCKF or MTH_FCNTL

#  define MTH__FCNTL
#  include <unistd.h>
#  include <fcntl.h>

#else

#  define MTH__ZERO

#  error " System not supported => File Guard module is Empty."
#endif




#ifdef MTH_FSOPEN
// When it exist a special open for shared file (as with Windows (and old OPENVMS).
// Currently, it is specific of WINDOWS O.S.
//

static int  FG_Open( FILE * &fp,
                     const char * path, const char * lb = 0 )
{
    // Open the file for exclusive write use (totaly exclusive use with _SH_DENYRW).
    fp = _fsopen( path, "wt", _SH_DENYWR );
    if (!fp)
        return (errno != EINVAL) ? -1 : 1; // Error for -1 or Locked file for +1.
    fprintf( fp, "%s Guard ID = %d\n", lb, GetCurrentProcessId() );
    fflush( fp );
    // The file stay Opened.
    return 0;
} // static int  FG_Open( const char * path ).


static void FG_Close( FILE * &fp )
{
    fclose( fp ); fp = NULL;
} // static void FG_Close().




#elif defined( MTH_FLOCK )

// This API is LINUX specific but Linux can use also lochf and fcntl.
// it would be preferable to have the file lock active for files accessed ...
// ...  over the network.
// It depends on versions of Linux kernel, NFS, Samba...

static int  FG_Open( FILE * &fp,
                     const char * path, const char * lb = 0 )
{
    int rc, ie;

    // Open the file for exclusive write use (totaly exclusive use with _SH_DENYRW).
    if (fp = fopen( path, "wr" )) {
        rc = flock( fileno( fp ), LOCK_EX|LOCK_NB );
        if (rc) {
            // Check for lock error or file locked.
            ie = (errno == EWOULDBLOCK) ? 1 : -1;
            fclose fp );
            fp = NULL;
        } else {
            // The file can be create/supershed :
            fprintf( fp, "%s ID = %d\n", lb, getpid() );
            fflush( fp );
            ie = 0; // Signal the file as opened.
            // the file stay opened.
        }
    } else  ie = -1; // fopen error -cannot open file.
    return ie;
} // static int  FG_Open( const char * path ).



static void FG_Close( FILE * &fp )
{
    // Closing the file causes the lock to be removed.
    // flock( fileno(fp), LOCK_UN );
    fclose( fp ); fp = NULL;
} // static void FG_Close().



#elif defined( MTH_LOCKF )


static int  FG_Open( FILE * &fp,
                     const char * path, const char * lb = 0 )
{
    int rc, ie;

    // Open the file for exclusive write use (totaly exclusive use with _SH_DENYRW).
    if (fp = fopen( path, "wr" )) {
        rc = lockf( fileno( fp ), F_TLOCK, 0 );
        if (rc) {
            // Check for lock error or file locked.
            ie = (errno == EACCES || errno == EAGAIN) ? 1 : -1;
            fclose( fp );
            fp = NULL;
        } else {
            // The file can be create/supershed :
            fprintf( fp, "%s ID = %d\n", lb, getpid() );
            fflush( fp );
            ie = 0; // Signal the file as opened.
            // the file stay opened.
        }
    } else  ie = -1; // fopen error -cannot open file.
    return ie;
} // static int  FG_Open( const char * path ).



static void FG_Close( FILE * &fp )
{
    // Closing the file causes the lock to be removed.
    // fseek( fp, 0, SEEK_SET );
    // lockf( fileno(fp), F_ULOCK, 0 );
    fclose( fp ); fp = NULL;
} // static void FG_Close().



#elif defined( MTH__FCNTL )
//   default to MTH_FCNTL


static int  FG_Open( FILE * &fp,
                     const char * path, const char * lb = 0 )
{
    int rc, ie;
    struct flock fl = {
        F_WRLCK,  // l_type   => Lock for Write.
        SEEK_SET, // l_whence => for the whole of file.
        0,        // l_start
        0         // l_len    => (0 = until eof).
    };

    // Open the file for exclusive write use (totaly exclusive use with _SH_DENYRW).
    if (fp = fopen( path, "wr" )) {
        rc = fcntl( fileno( fp ), F_SETLK, &fl );
        if (rc) {
            // Check for lock error or file locked.
            ie = (errno == EACCES || errno == EAGAIN) ? 1 : -1;
            fclose fp );
            fp = NULL;
        } else {
            // The file can be create/supershed :
            fprintf( fp, "%s ID = %d\n", lb, getpid() );
            fflush( fp );
            ie = 0; // Signal the file as opened.
            // the file stay opened.
        }
    } else  ie = -1; // fopen error -cannot open file.
    return ie;
} // static int  FG_Open( const char * path ).



static void FG_Close( FILE * &fp )
{
    // Closing the file causes the lock to be removed.
    // fseek( fp, 0, SEEK_SET );
    // fcntl( fileno(fp), F_UNLCK, &fl );
    fclose( fp ); fp = NULL;
} // static void FG_Close().


#else

static int  FG_Open( FILE * &fp,
                     const char * path, const char * lb = 0 )
{ return -1; }

static void FG_Close( FILE * &fp ) { return -1; }

#endif


//
//  The Class FileLock functions :
//

int  FileLock::OpenL(  const char * pa, const char * lb )
{
    if (lb) {
        if (md_) delete[] md_;
        md_ = strdup( lb );
    }
    return FG_Open( fp_, pa, md_ );
} // int  FileLock::OpenL(  const char * pa, const char * lb ).


void FileLock::CloseL()
{
    FG_Close( fp_ );
} // void FileLock::~CloseL().


int  FileLock::Write( const char * fmt, ... )
{
    va_list ap;
    int     nv;

    va_start( ap, fmt );
    nv = ::vfprintf( fp_, fmt, ap );
    va_end( ap );
    fflush( fp_ );
    return nv;
} // void FileLock::Write( const char * frm, ... ).



int  FileLock::Read( const char * fmt, ... )
{
    va_list ap;
    int     nv;

    va_start( ap, fmt );
    nv = ::vfscanf( fp_, fmt, ap );
    va_end( ap );
    return nv;
} // void FileLock::Read( const char * frm, ... ).



//
// end of <PW-Id>.
//
