//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// Module of service routines/objects.
//

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <FL/fl_utf8.h>         // To manipulate utf8 strings.

#include "Service_Util.h"


#define VSTR_ALLOC_STEP 16
#define PATH_ALLOC_STEP 64
#define TABL_ALLOC_STEP 16
#define MAX_LOG_LENGTH  32

//
// Functions of <string.h> missing on windows.
//

#ifdef WIN32

const char * rindex( const char * s, char c )
{
    int l;

    if (s && s[0]) {
        l = strlen( s ) - 1;
        while (l >= 0 && s[l] != c) --l;
        if (l >= 0) return s + l;
    }
    return 0;
} // SHould be in <string.h>


#endif


//
// Functions for Temporary string allocations/free class VarString.
//

char * VarString::Size( int size )
{   char *nst_;
    use_ = size + 1;            // Add one for the final null character.
    if (all_ < use_) {          // We must increase the string allocation.
        all_ = (use_ + VSTR_ALLOC_STEP)&~(VSTR_ALLOC_STEP-1);
        nst_ = new char[all_];  // Allocate the new string.
        // Copy the old string in the new one when required.
        if (cop_&&str_&&str_[0]) strcpy( nst_, str_ );
        else nst_[0] = 0;
        if (str_) delete[] str_;// Free old string.
        str_ = nst_;
    }
    return str_;
} // void VarString::Allocate( int size ).



//
// **********************************************************************************
//


//
// Class FilePath to manage a dynamic file path.
//


FilePath::~FilePath()
{
    if (path_) delete[] path_;
    if (stack_) delete [] stack_;
    path_ = 0; stack_ = 0;
} // FilePath::~FilePath().



int FilePath::Set( const char * name, int dfl )
//
// Set a string in *PathString to be the path of the new file of directory.
// SetFilePath return the old PathString length to be reset to old value by
// the routine RemFilePath( int ).
// When the argument name is NULL the PathString is preserved.
// The argument dfl is used to set or remove a terminated Directory separator
// in all case, (name argument provided of NULL).
//
{
    int len_, oln_, usz_;
    int            * tm_;
    
    oln_ = length_;
    if (name) {
        len_ = strlen( name );
        usz_ = len_ + length_ + 2;
        if (allsiz_ < usz_) {
            // Allocated size round up to a multiple of PATH_ALLOC_STEP.
            allsiz_ = (usz_ + PATH_ALLOC_STEP)&~(PATH_ALLOC_STEP-1); 
            char * tmp_ = new char[allsiz_];
            if (path_) {
                strcpy( tmp_, path_ );
                delete[] path_;
            }
            path_ = tmp_;
        }
        usz_ = length_ + len_;
        strcpy( path_ + oln_, name );
    
        sp_++;
        if (sl_ <= sp_) {
            sl_ = (sl_ + TABL_ALLOC_STEP)&~(TABL_ALLOC_STEP - 1);
            tm_ = new int[sl_];
            if (sp_ > 0) {
                for(int ii = 0; ii < sp_; ii++) tm_[ii] = stack_[ii];
                delete[] stack_;
            }
            stack_ = tm_; tm_ = 0;
        }
        stack_[sp_] = oln_;
    } else usz_ = length_;

    if (usz_ > 0) {
        if (path_[usz_-1]!=DIR_SEPAR) {
            if (dfl) {
                path_[usz_++] = DIR_SEPAR;
                path_[usz_] = 0;
            }
        } else {
            if (!dfl) path_[--usz_] = 0;
        }
    }
    length_ = usz_;
   return oln_;
} // int FilePath::Set( const char * name, int dfl ).



int FilePath::Rem()
{
    int ie = 0;

    if (stack_) {
        if (sp_ >= 0) length_ = stack_[sp_--];
        else ie = -1;
    }
    path_[length_] = 0;
    return ie;
} // int FilePath::Rem( int vl ).



void FilePath::Clear()
{
    if (path_) path_[0] = 0;
    length_ =  0;
    sp_     = -1;
} // void FilePath::Clear().



void FilePath::BuiltIter( const char * pt )
{
    const char   * cp, * vp;
    char  ch, lg[MAX_LOG_LENGTH];
    int                  ii;

    cp = strchr( pt, DEV_SEPAR );       // Look for ':' (return 0 if not found).
    if (cp) {                           // When found ...
        vp = cp -  1;                   // ... vp -> last character of device name.
        ii = cp - pt;                   // ii = size of device name.
        if (ii >= MAX_LOG_LENGTH) exit( 2 );
        lg[ii]  =  0;                   // Copying the device name in major case.
        while (ii--) {
            ch =*(vp--);
            if (ch >= 'a' && ch <= 'z') ch += 'A' - 'a';
            lg[ii] = ch;
        }
        vp = getenv( lg );              // If it is a logical device name, get it corresponding path.
        if (vp) {                       // We hae the logical device translation.
            BuiltIter( vp );            // Loop for more internal logical reference ...
            pt = cp + 1;                // ... and skip the device separator.
        }
        Set( NULL, 1 );
    }
    if (pt[0]) Set( pt, 0 );            // Put in path the traling path specifications.
} // void FilePath::BuiltIter( const char * pt ).




int  FilePath::PathSearch( const char * PathList, const char * fname )
// Look for the first existing path in Pathlist (with optional filename).
// The reulting path is keep in the FilePath object and the the return value
// is the path entry index (starting from 1) when found or else 0.
//
{
    const char        *cu, *lm;
    char             buf[1024];
    int             ii, jj, ne;

    cu = PathList;                              // Start scan from the begin of Path List.
    jj = ne = 0;
    do {                                        // Loop on all Path of the path list.
        jj++;
        Clear();                                // Erase any previous path.
        ii = 0;
        lm = strchr( cu, PAT_SEPAR );           // Look for the first/next path separator.
        if (!lm) lm = cu + strlen( cu );        // If not found, this path is the last one.
        while (cu != lm) buf[ii++] = *(cu++);   // We copy this path in our buffer (no overflow checking).
        buf[ii] = 0;
        BuiltIter( buf );                       // Get the translation of any logical device reference in this path.
        if (fname) {                            // When an additional filename is specified ...
            Set( NULL, 1 );                     // ... we append it in the path.
            Set( fname, 0 );
        }
        // We check if the file or directory is existing.
        if (!access( Path(), F_OK)) { ne = jj; break; }
        // When it is not existing, skip the path separator (when not last path) ...
        if (*cu == PAT_SEPAR) cu++;             // ... and loop for the next path.
    } while (*cu);
    return ne;                                  // Return the path index (starting from 1).
} // const char * FilePath::PathSearch( const char * pathList, const char * fname ).





// --------------------------------------------------------------------------------


//
// Object table class routines to manage a tabel of undefined objects.
//

int  ObjTable::AddObj( ObjPtr p )
{
    ObjPtr *tm;
    int   size;

    size = use_ + 1;
    if (size >= all_) {
        size = (size + TABL_ALLOC_STEP)&~(TABL_ALLOC_STEP - 1);
        tm = new ObjPtr[size];
        if (use_)
            for(int ii = 0; ii < use_; ii++) tm[ii] = tab_[ii];
        if (tab_) delete[] tab_;
        tab_ =   tm;
        all_= size;
    }
    tab_[use_++] = p;
    return use_;
} // void ObjTable::AddObj( ObjPtr p ).



ObjPtr * ObjTable::GetObjs( int shf )
{
    if ((shf >= 0)&&(shf < use_))
        return (ObjPtr*)((char*)tab_ + shf );
    else
        return NULL;
} // ObjPtr * ObjTable::GetObjs( int shf )



void ObjTable::Tamp( int idx )
{
    int  i,  j;

    if ((idx < 0)||(idx >= use_)) return;

    if (tab_) {
        i = j = idx;
        while (i < use_) {
            if (tab_[i]) {
                if (j != i) tab_[j] = tab_[i];
                i++; j++;
            } else i++;
        }
        use_ = j;
    }
} // void ObjTable::Tamp().



void ObjTable::RemObj( int idx )
{
    if ((idx >= 0)&&(idx < use_)) {
        tab_[idx] = NULL;
        this->Tamp( idx );
    }
} // void ObjTable::RemObj( int idx ).



void ObjTable::RemObj( ObjPtr p )
{
    int idx = 0;

    while ((idx < use_)&&(tab_[idx] != p)) idx++;
    if (idx < use_) {
        tab_[idx] = NULL;
        this->Tamp( idx );
    }
} // void ObjTable::RemObj( ObjPtr p ).



//
// end of <PW-Id>.
//
