//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module to read any configuration file.
//


#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <math.h>

#define SETUP_MANAGER 1

#include "SetUpMan.h"



// Define the maximum string size.
#define   MAX_STR           256


// Define the directory separator (defined here to be independent module).
#if (!defined( _WIN32)) || defined(__CYGWIN__)
//   Unix/Linux/Cygwin Directory separator.
# define DIR_SEPAR         '/'
#else
//   Windows Directory separator.
# define DIR_SEPAR         '\\'
#endif
#define  DEV_SEPAR         ':'

// Define the greatest 32 bits integer value.
#define  MAXINT         2147483647
#define  RMAXINT      2147483647.0

// Minimum limit to change of line when write a SetUp File.
#define  MAXLSFSZ     120



//
// SetUp descrition structure definitions.
//

typedef unsigned int UInt32;


// Symbols definitions.
// the ordering must same that for SetUpSym.
typedef enum { identsy = 0,
               integersy, floatsy, stringsy,
               listsy, queuesy,
               ct_stringsy, ct_integersy, ct_floatsy,
               eofsy, eolnsy,
               colonsy, commasy, semicolonsy, equalsy,
               lparensy, rparensy,
               separatorsy
             } symbol;

             
// Define the syntaxic element record type.
typedef struct { symbol             sy; // Type of item,
                 SetUpIde const * ptid; // Pointer to identifier,
                 float            fltv; // Read float value,
                 int              intv, // Read integer value,
                                  strl; //...and related length,
                 char strv[MAXLINESZ+1];// String/Identifier value.
} item;

// Define Union to store in same location Variable address. 
typedef union {  // Union for pointer and value equivalence.
                 void               *ad; // void address.
                 SetUpDsc           *al; // Descriptor address.
                 char              **as; // String Address.
                 int                *ai; // Integer Address.
                 float              *af; // Floating number address.
} EqObj;    



               
// Define the various categories of ASCII characters.
typedef enum   {        
    alpha,digit,period,comma,semicol,colon,
    addo,subo,mulo,divo,powo,
    quot,lpar,rpar,equo,oth,eofo,eolno,spacech,baseo,
    clto,cgto,ando,commento,lbra,rbra
} chcat;


int     BadSetUpFile    =    0; // Global flag for SetFile with syntax Error.
               
// Character attribute array.
static const chcat  attchr[96] =
{ // _oth_ for unused characters.
    spacech,  commento, quot,     baseo,    alpha,    baseo,    ando,     quot,
  // ' '       '!'       '"'       '#'       '$'       '%'       '&'       '\''
    lpar,     rpar,     mulo,     addo,     comma,    subo,     period,   divo,
  // '('       ')'       '*'       '+'       ','       '-'       '.'       '/'
    digit,    digit,    digit,    digit,    digit,    digit,    digit,    digit,
  // '0'       '1'       '2'       '3'       '4'       '5'       '6'       '7'
    digit,    digit,    colon,    semicol,  clto,     equo,     cgto,     oth,
  // '8'       '9'       ':'       ';'       '<'       '='       '>'       '?'
    oth,      alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
  // '@'       'A'       'B'       'C'       'D'       'E'       'F'       'G'
    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
  // 'H'       'I'       'J'       'K'       'L'       'M'       'N'       'O'
    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
  // 'P'       'Q'       'R'       'S'       'T'       'U'       'V'       'W'
    alpha,    alpha,    alpha,    lbra,     oth,      rbra,     powo,     alpha,
  // 'X'       'Y'       'Z'       '['       '\\'      ']'       '^'       '_'
    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
  // '`'       'a'       'b'       'c'       'd'       'e'       'f'       'g'
    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
  // 'h'       'i'       'j'       'k'       'l'       'm'       'n'       'o'
    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,    alpha,
  // 'p'       'q'       'r'       's'       't'       'u'       'v'       'w'
    alpha,    alpha,    alpha,    oth,      oth,      oth,      oth,      oth
  // 'x'       'y'       'z'       '{'       '|'       '}'       '~'       '?'
};


static int  binstring  = false; // Boolean to flag the string interpretation of readden characters.
static int  cter       = false; // Boolean to flag the comment interpretation of readden characters.
static int            ch, cmaj; // Input character as used by InSymbol.

static chcat    categ  = eolno; // Current character kind.

static int                ierr; // Error code for SetUp file syntax error.
static item                sym; // The syntax item element as readden by InSymbol. 
static FILE        *        fp; // file varible used to read or write the SetUp files.
static const char * SetUpPath = NULL; // Complete Path of the setup File and ...

static const SetUpIde * IdeTable = NULL; // Current ident table used by SearchId function.

// Variable to keep address of the User Error Display function.
static SetUpErrDisplay *UserErrDisplay = NULL;

static int                    SetUpCol; // To limit the size of SetUp file line.

//
// The SetUpMan manager functions.
//


static void SignalError( int ie, char const *msg, const char *id )
// Procedure to manage the syntax errors in SetUp file(s).
{
    int     i,  j,  l, nc;
    static char  buf[512];
    char             * pm;

    nc = snprintf( buf, 512, "SetUpMan%s syntax error %d in setup file :\n\t\"%s\"\n",
                   (ie < 0)? " Fatal" : " ", abs( ie ), SetUpPath );
    pm = buf + nc;              // Now pm is address of temporary end of the string buffer.
    i = l = strlen( msg );
    while ((i>0)&&(msg[i]!='`')) i--;   // Locate the id position in msg.
    if (i) {
        for(j=0;j<i;j++) *(pm++) = msg[j];
        for(j=0;j<strlen(id);j++) *(pm++) = id[j];
        for(j=i+1;j<=l;j++) *(pm++) = msg[j];
        *(pm) = 0;
    }

    if (ie < 0) BadSetUpFile = -1;

    if (UserErrDisplay) UserErrDisplay( buf );
                   else fprintf( stderr, "%s\n", buf );
    if (BadSetUpFile) exit( 1 );
} // void SignalError( int ie, char const *msg, const char *id ).



void SetUpErrDisplayInstall( SetUpErrDisplay *userf )
// Install or UnInstall (with NULL argument) a Display error message function.
//
{
    UserErrDisplay = userf;
} // void SetUpErrDisplayInstall( ErrMsgDisplay *userf ).



static int Match( const char *s1, const char *s2, int l1, int l2 )
{  /*
    * To compare two identifier names
    * the result value is :
    *   0 if s1 and s2 match,
    *   1 if s1 > s2 and -1 if s1 < s2
    */

    int i, j, bl;

    if (l1 > l2) j = l1;
            else j = l2;
    i = 1;
    bl = true;

    while ((i <= j) && bl) {
        bl = (s1[i] == s2[i]);
        if (bl) i = i + 1;
    }

    if (bl) {
        if (l1 == l2) return 0;
              else if (l1 > l2) return  1;
                           else return -1;
    } else
        if (s1[i] > s2[i]) return  1;
                      else return -1;
} // static int Match( const char *s1, const char *s2, int l1, int l2 ).



static SetUpIde const * SearchId( SetUpIde const idt[], const char *nam, int len )
// SearchId: to search an identifier.
{
    int                      i;
    char const            * nm;
    SetUpIde const * id = NULL;

    i =      0;
    do {
      id = &idt[i++];
      nm = id->idname;
    } while (nm&&(Match( nm, nam, strlen( nm ), len ) != 0));

    if (!nm) {
        SignalError( 11, "SeId: Uknown Identifier <`> in setting file.", nam );
        return NULL;
    }
    return id;
} // static SetUpIde * SearchId( SetUpIde idt[], char *nam, int len ).



static void GetDChar( void )
/* To get one character in the current line
   and set these characteristics.
*/
{
    ch = getc( fp );
    if (ch == EOF) {
          ch = ' '; cmaj = ch; categ = eofo;
    } else if ((ch == 0) || ((ch == '\n' /* LINE FEED */)&&cter)) {
        categ = eolno; ch = 0;
    } else {
        if (ch < ' ') ch = ' ';
        cmaj = ch;
        if ((ch >= 'a')&&(ch <= 'z')) cmaj = ch - 32;
                                 else cmaj = ch;
        if (cmaj > '_') cmaj = cmaj - 32;  /* convert lower to upper case */
        if (ch > 127) categ = oth;
                 else categ = attchr[ch - ' '];
    }
} // static void GetDChar( void ).



static void Comment( char cs )
{ // READ A COMMENT TO PROCEED IN INPUT STREAM.

    cter = true;
    GetDChar();
    while ((ch != cs) && (categ != eofo)) {
        GetDChar();
    }
    cter = false;
    cmaj = ' ';
    categ = spacech;
    ch = ' ';     // set the space state.
} // static void Comment( char cs ).



static void NextCh( void )
{ /* get a character from the input stream
   * with comment elimination
   * NextCh
   */

    GetDChar();
    if (!binstring)
        if (ch == '{') Comment( '}' );
                  else if (ch == '!') Comment( 0 );
} // static void NextCh( void ).



static void SkipSpace( void )
{ // To skip unsignificative space and comment.
    do {
        NextCh();
    } while ((ch <= ' ') /* && (categ != eolno) */ && (categ != eofo));
} // static void SkipSpace( void ).



static void InSymbol( void )
{   /* InSymbol :
     * Get a symbol in the source input stream and decode it.
     */
    const int ten = 10,   one =  1;
    int   bneg,  bint,  ivl,  base;
    double  rdig, rexp, rfac, rval;
    char                       * s;

    if ((ch == ' ') && (categ != eolno) && (categ != eofo)) SkipSpace();
    sym.ptid = NULL;
    switch (categ) {
        case addo:      // Negate sign.
        case subo:      // Addition sign.
        case digit:     // '0'...'9'
        case period:    // '.'...number */
            bneg = 0;
            if (categ==subo) { bneg = 1; NextCh(); }
            if (categ==addo) NextCh();
            if ((categ!=digit)&&(categ!=period)) {
                SignalError( -12, "InSy: illegal use of arithmetic sign in setting file", NULL );
            }
            // Assuming integer until shown other.
            ierr = false;
            rval = 0.0;
            rexp = ten;
            rfac = one;
            sym.sy = integersy;
            while (categ == digit) {
                rdig = ch - '0';
                NextCh();
                rval = rval * ten + rdig;
            }
            if (categ == period) {
                NextCh();
                sym.sy = floatsy;
                while (categ == digit) {
                    rdig = ch - '0';
                    NextCh();
                    rfac = rfac / ten;
                    rval = rval + rfac * rdig;
                }
            }
            if ((cmaj == 'e') || (cmaj == 'E')) {
                sym.sy = floatsy;
                NextCh();
                if ((ch == '+') || (ch == '-')) {
                    if (ch == '-') rexp = one/rexp;
                    NextCh();
                }
                ivl = 0;
                while (categ == digit) {
                    ivl = ivl * 10 + ( ch - '0');
                    NextCh();
                }
                if (ivl > 38) {
                    SignalError( -11, "InSy: EXPOSANT > 38 in setting file", NULL );
                    ierr = true;
                }
                rfac = one;
                while (ivl != 0) {
                    if ((ivl % 2) == 1) {       // ODD ivl */
                        ivl = ivl - 1;
                        rfac = rfac * rexp;
                    } else {                    // Even ivl
                        ivl = ivl / 2;          // ivl div 2
                        rexp = rexp * rexp;     // sqr(rexp)
                    }
                }
                rval = rval * rfac;
            }
            if (bneg) rval = -rval;
            if ((sym.sy == integersy) &&
                (rval <= RMAXINT) && (rval >= -RMAXINT-1.0)) {
                sym.fltv = floor( rval + 0.5 ); // ROUND(rval
                sym.intv = (int) sym.fltv;
            } else {
                sym.intv = 0;
                sym.sy = floatsy;
            }
            sym.fltv = rval;
        break;

        case alpha:     // 'a'...'z'....IDENTIFIER or KEYWORD or OPERATOR
                        // We must have MAXIDSIZE >= MAXLINESZ
            sym.strv[0] =  0;
            sym.strl    =  0;
            ivl = 0;
            while ((categ == alpha) || (categ == digit) || (categ == period)) {
                if (ivl < MAXIDSIZE) sym.strv[ivl] = cmaj;
                ivl = ivl + 1;
                NextCh();
            }
            if (ivl >= MAXIDSIZE) ivl = MAXIDSIZE;
            sym.strv[ivl] =        0;
            sym.strl      =      ivl;
            sym.sy        =  identsy;
            if (sym.ptid = SearchId( IdeTable, sym.strv, sym.strl ))
                switch (sym.ptid->vdescr.vtype) {
                    case SetUpCtI:
                        sym.sy   = integersy;
                        sym.intv = ((int *)(sym.ptid->vdescr.vaddr)) [0];
                    break;
                    case SetUpCtF:
                        sym.sy   = floatsy;
                        sym.fltv = ((float *)(sym.ptid->vdescr.vaddr)) [0];
                    break;
                    case SetUpCtS:
                        sym.sy   = stringsy;
                        s = ((char **)(sym.ptid->vdescr.vaddr)) [0];
                        sym.strl = 0;
                        if (s&&s[0])
                            while (s[sym.strl]) sym.strv[sym.strl++] = *(s++);
                    break;
                    default: ;          // Nothing to do.
                }
            else {
                SignalError( 12, "InSy:  Application Setup syntax error: unknown identifier \"`\" => ignore it.\n", sym.strv );
                // sym.sy = eofsy;         // To stop the setUpRead process.
            }
        break;

        case baseo: // '#' or '%' --> IT IS A BASE
            ierr = false;
            sym.sy = integersy;
            base = 10;
            if (ch == '#') {
                NextCh();
                base = 0;
                while (categ == digit) {
                    base = base * 10 + ( ch - '0');
                    NextCh();
                }
                if (ch != '#') {
                    SignalError( -13, "InSy: Bad Base specification in setting file", NULL );
                    ierr = true;
                }
            } else if (ch == '%') {
                base = 10;
                NextCh();
                switch(ch) {
                    case 'X':
                    case 'x':
                    case 'H':
                    case 'h':
                        base = 16;
                    break;

                    case 'D':
                    case 'd':
                        base = 10;
                    break;

                    case 'O':
                    case 'o':
                        base =  8;
                    break;

                    case 'B':
                    case 'b':
                        base =  2;
                    break;

                    default: // base = 10
                    break;
                }
            }
            if (!ierr) {
                NextCh();
                ivl = 0;
                while (ch > ' ') {
                    if ((ch >= '0') && (ch <= '9')) ch = ch - '0';
                    else if ((ch >= 'a') && (ch <= 'f')) ch = ch - 'a' + 10;
                    else if ((ch >= 'A') && (ch <= 'F')) ch = ch - 'A' + 10;
                    if (ch < base) {
                        ivl = base*ivl + ch;
                        NextCh();
                    } else ch = 0;
                }
                sym.intv = ivl;
                sym.fltv = sym.intv;
            }
        break;

        case equo:  // '='
            sym.sy = equalsy;
            NextCh();
        break;

        case colon: // ':'
            sym.sy = colonsy;
            NextCh();
        break;

        case comma: // ','
            sym.sy = commasy;
            NextCh();
        break;

        case semicol: // ';'
            sym.sy = semicolonsy;
            NextCh();
        break;

        case quot:          // Constructor of string
            bint = true;
            sym.strv[0] =  0;
            sym.strl    =  0;
            ivl         =  0;
            binstring = true;
            while (bint) {
                NextCh();
                if (categ == eolno) NextCh();   // To skip any line change
                if (categ == eofo) bint = false;
                if (ch == '\'') {
                    NextCh();
                    if (ch == '\'') sym.strv[ivl] = ch;
                               else bint = false;
                } else {
                    sym.strv[ivl] = ch;
                    if (bint && (ivl < MAXLINESZ))
                        ivl = ivl + 1;
                    else if (bint) {
                        SignalError( -14, "InSy: Line too long in setting file", NULL );
                        bint = false;
                    }
                }
            }
            sym.strv[ivl] = 0;
            binstring = false;
            sym.strl  =   ivl;
            sym.sy = stringsy;
        break;

        case lpar:
            sym.sy = lparensy;
            NextCh();
        break;

        case rpar:
            sym.sy = rparensy;
            NextCh();
        break;
        

        case eolno:
            sym.sy = eolnsy;
            NextCh();
        break;

        case eofo:
            sym.sy = eofsy;
        break;

        default:    // separator.
            sym.sy = separatorsy;
            NextCh();
        break;
    } // switch (categ).

#ifdef DEBUG

    switch (sym.sy) {
        case identsy:     fprintf( fmsg, " Read key identifier (%d) \"%s\"\n", sym.strl, sym.strv ); break;
        case integersy:   fprintf( fmsg, " Read integer %d\n", sym.intv ); break;
        case floatsy:     fprintf( fmsg, " Read float %d\n", sym.fltv ); break;
        case stringsy:    fprintf( fmsg, " Read string (%d) \"%s\"\n", sym.strl, sym.strv ); break;
        case colonsy:     fprintf( fmsg, " Read \":\"\n" ); break;
        case commasy:     fprintf( fmsg, " Read \",\"\n" ); break;
        case semicolonsy: fprintf( fmsg, " Read \";\"\n" ); break;
        case equalsy:     fprintf( fmsg, " Read \"=\"\n" ); break;
        case separatorsy: fprintf( fmsg, " Read other separator.\n" ); break;
        case eolnsy:      fprintf( fmsg, " Read EOLN\n" ); break;
        case eofsy:       fprintf( fmsg, " Read EOF\n" ); break;
        default: fprintf( fmsg, " read code = %d > %d\n", (int) sym.sy, (int) identsy ); break;
    }
    fflush( fmsg );

#endif

} // static void InSymbol( void ).



static void SkipToSymbol( symbol syv )
{
    while ((sym.sy != eofsy) && (sym.sy != syv)) InSymbol();
} // static void SkipToSymbol( symbol syv ).



static void ErrAndSkip( int ie, const char *msg, const char * id, symbol syv )
{
    SignalError( ie, msg, id );
    SkipToSymbol( syv );
} // static void ErrAndSkip( int ie, const char *msg, const char * id, symbol syv ).


/*
static QuElm * QuLocate( QuHeader *hde, int idx )
// If idx > 0, QuLocate() return the Queue element of the idx index.
// if idx < 0, we return the first element of the queue,
// When ide = 0, QuLocate() return the next element of the queue
// starting from the previously located element. When to try to
// get a non existant element, QuLocate() return a NULL pointer.
// The index (user with ide > 0) are defined by the SetUpRead()
// or SetUpQAppend<typ>() functions by an integer counter of new elements.
// If the function SetUpQRemove() is used, some element can be missing in
// the queue and the call of QuLocate() if ide>0 can be failed (return NULL)
// because the required element was removed. Consequently, Qlocate( <Queue>, 1 )
// and Qulocate( <Queue>, -1 ) are not equivalent. the calls Qlocate with ide <=0
// are used to scan the que in the Que creation ordering. and if Qulocate() with
// ide = 0 follows a previous call with idx > 0, the scan continue after
// to start from the element specified by idx > 0.
// the functions SetUpQGet<typ>() call QuLocate() if its argument idx to find the
// specified element and return its value.
//
{
    QuElm *elm;
    
    if (idx>0) { // search the specified element.
        elm = hde->first;
        while ((elm)&&(elm->idnb<idx)) elm = elm->next;
        if ((elm)&&(elm->idnb==idx)) hde->curr = elm;
                                else elm = NULL;
    } else {
        if (idx<0) {
            elm = hde->curr = hde->first;
        } else {
            elm = hde->curr->next;
            hde->curr = elm;
        }
    }
    return elm;
} // static QuElm * QuLocate( QuHeader *hde, int idx ).



static QuElm * QuAppend( QuHeader * hde )
{
    QuElm *elm = (QuElm*)malloc( sizeof( QuElm ) );
    if (hde->last) hde->last->next = elm;
              else hde->first = elm;
    hde->curr =       elm;
    elm->prev = hde->last;
    hde->last =       elm;
    elm->next =      NULL;
    elm->idnb = ++(hde->count);
    elm->data =      NULL;
    elm->obj.s =     NULL;
    return elm;
} // static QuElm * QuAppend( QuHeader * hde ).
*/


static int CheckSymbol( symbol wrd, const char * vnam, int skp )
// To check for agrement between the expected symbol <wrd>  and the read symbol..
// If (skp&1) (is odd skp), CheckSymbol() skip to the next syntax unit?
// If (skip&), when wrd = commasy, CheckSymbol() accept a semicolon(sy), else
// it accept a right parenthesys. In these both cases, the (semicolon or the
// right parenthesys) symbol is never skipped but comma can be skipped if (skp&1).
//
{
    char *msg  = NULL;
    int   ier  =    0;
    
    static char
        msg01[] = "An \"=\" for assignation of <`> was expected\n",
        msg02[] = "A left parenthesys \"(\" was expected after \"` =\"\n",
        msg03[] = "A right parenthesys \")\" was expected after \"` =\"\n",
        msg04[] = "A semicolon was expected after \"` =\"\n",
        msg05[] = "A comma or right parenthesys was expected after \"` =\"\n",
        msg06[] = "A comma or semicolon was expected after \"` =\"\n",
        msg07[] = "An integer value was expected for <`>\n",
        msg08[] = "A floating value was expected for <`>\n",
        msg09[] = "A string value was expected for <`>\n",
        msg10[] = "A colon symbol was expected after an array size value for <`>\n",
        msg11[] = "Illegal sign + or - not before a number for symbol <`>\n";

    if (sym.sy==wrd) {
        if (skp&1) InSymbol();
    } else {
        switch (wrd) {
            case equalsy: 
                msg = msg01; ier =  1;
            break;
            case lparensy:
                msg = msg02; ier =  2;
            break;
            case rparensy:
                msg = msg03; ier =  3;
            break;
            case semicolonsy:
                msg = msg04; ier =  4;
            break;
            case commasy:
                if (skp&2) {
                    if (sym.sy!=semicolonsy) { msg = msg06; ier =  6; }
                }
                else
                    if (sym.sy!=rparensy) { msg = msg05; ier =  5; }
            break;
            case integersy:
                if (sym.sy==floatsy) {
                    if (sym.fltv >= 0.0) sym.intv = (int) (sym.fltv + 0.5);
                                    else sym.intv = (int) (sym.fltv - 0.5);
                    sym.sy = integersy;
                }
                if (sym.sy!=integersy) { msg = msg07; ier =  7; }
            break;
            case floatsy:
                if (sym.sy==integersy) {
                    // Implicit convert of integer value to float
                    sym.fltv = sym.intv; sym.sy = floatsy;
                }
                if (sym.sy!=floatsy) { msg = msg08; ier =  8; }
            break;
            case stringsy:
                msg = msg09; ier =  9;
            break;
            case colonsy:
                msg = msg10; ier = 10;
            break;
            default: ;
        }
        if (ier>0) {
            ErrAndSkip( ier, msg, vnam, semicolonsy );
            BadSetUpFile = -1;
        }
    }
    return ier;
} // static int CheckSymbol( symbol wrd, const char * vnam, int skp ).



static void SetParmDynTab( const char *vnam, SetUpSym vty, void *obj );

static void SetParmValue( const char *vnam, SetUpSym vty, int siz, void *obj, int flg )
// To read a single variable or a single array,
// and a pseudo record element that can be single object or array.
// flg = 0/1 for fixed/dynamic size array.
//
{
    EqObj  ptr;    
    int    cnt;
    char  *str;
    
    cnt    =   0;
    ptr.ad = obj;
    if (flg||siz>1) CheckSymbol( lparensy, vnam, 1 ); 
    while (cnt < siz) {
        // All unassigned, we keep old values.
        if ((sym.sy==semicolonsy)||(sym.sy==lparensy)) break;
        if (sym.sy!=commasy) {
            switch (vty) {
                case SetUpInt: // Assign an integer value.
                    CheckSymbol( integersy, vnam, 0 );
                    *(ptr.ai++) = sym.intv;
                break;
                case SetUpFlt: // Assign a floating value.
                    CheckSymbol( floatsy, vnam, 0 );
                    *(ptr.af++) = sym.fltv;
                break;
                case SetUpStr: // Assign a string value.
                    if (!CheckSymbol( stringsy, vnam, 0 )) {
                        str = *(ptr.as);        // For a string, we must ...
                        if (str) free( str );   // free the old string before to load a new one.
                        if (sym.strl > 0) str = strdup(  sym.strv );
                                     else str = NULL;
//                      str = (char *) malloc( sym.strl + 1 );
//                      for (int ii = 0; ii <= sym.strl; ii++ ) str[ii] = sym.strv[ii];
                        *(ptr.as++) = str;
                    }
                break;
                default: ;
            }
            InSymbol();         // Gobble up the value.
        } else {// else for if (sym.sy!=commasy) ... .
            // We must skip the missing field.
            switch (vty) {
                case  SetUpInt: ptr.ai++; break;
                case  SetUpFlt: ptr.af++; break;
                case  SetUpStr: ptr.as++; break;
                default: ;
            }
        }
        if (++cnt < siz) CheckSymbol( commasy, vnam, 1 );
    }
    if (flg||siz>1) CheckSymbol( rparensy, vnam, 1 );
} // static void SetParmValue( const char *vnam, SetUpSym vty, int siz, void *obj, int flg ).



static void SetParmDynTab( const char *vnam, SetUpSym vty, void *obj )
{
    SetUpSym       nty;
    SetUpDynTab  * dst;
    int           size;
    EqObj          ptr;

    CheckSymbol( integersy, vnam, 0 );
    size = sym.intv;          // Get the array size (in number of elements).
    InSymbol();               // Gobble up the size.
    dst = (SetUpDynTab*)obj;
    dst->size = size;         // Put the total number of elements.
    switch (vty) {
        case SetUpDIn:        // Table of integer :
            if (size) dst->tab.i = ptr.ai = new int[size];
                 else dst->tab.i = NULL;
            nty = SetUpInt;
        break;
        case SetUpDFl:        // Table of float :
            if (size) dst->tab.f = ptr.af = new float[size];
                 else dst->tab.f = NULL;
            nty = SetUpFlt;
        break;
        case SetUpDSt:
            if (size) {        // Table of string :
                dst->tab.s = ptr.as = new char *[size];
                for(int i = 0; i < size; i++) dst->tab.s[i] = NULL;
            } else dst->tab.s = NULL;
            nty = SetUpStr;
        break;
        default: ;
    }
    if (size) SetParmValue( vnam, nty, size, ptr.ad, 1 );

} // static void SetParmDynTab( const char *vnam, SetUpSym vty, void *obj )



static void LiSetParm( const char *vnam, SetUpDsc *dsc )
{
    SetUpDsc * fld;
    
    fld = dsc;
    CheckSymbol( lparensy, vnam, 1 );           // Gobble up the left parenthesys.
    while (fld&&fld->vtype!=SetUpEof) {         // Loop on all pseudo record field.
        // Deposite all specified value in each target field (single value or array).
        SetParmValue( vnam, fld->vtype, fld->vsize, fld->vaddr, 0 );
        if (sym.sy==rparensy) break;
        fld++;                                   // Skip to the next field.
        if (fld&&fld->vtype!=SetUpEof)           // If it is not the end of field list, ...
          CheckSymbol( commasy, vnam, 1 );       // ... we gobble up any comma that separ each field.
    }
    CheckSymbol( rparensy, vnam, 1 );            // Check and gooble up the final right parenthesys
} // static void LiSetParm( const char *vnam, SetUpDsc *dsc ).



static void ParamSetting( const char * fnam, const SetUpIde * idetab )
// To read each identifier described in the idetab from the current setup file.
{
    const SetUpIde       * ide;
    SetUpSym               vty;
    int                    siz;
    void                 * obj;
    const char          * vnam;

    if (idetab) {
        IdeTable = idetab;      // Get the identifier table origine for SearchId function. 
        ch    = ' ';            // Init the InSymbol environment.
        cmaj  = ' ';
        categ = spacech;
        InSymbol();                     // Get the first syntax unit.
        while (sym.sy!=eofsy) {         // Loop until the end of setup file.
            if (sym.sy==identsy) {      // We have found a known keyword.
                ide = sym.ptid;         // Get the identifier address (when find by InSymbol).
                if (ide) {
                    vnam = ide->idname; // Keep the location of SetUp Identifier name.
                    InSymbol();         // Gobble up the identifier and equal operator.
                    CheckSymbol( equalsy, vnam, 1 );
                    vty = ide->vdescr.vtype;  // Get the Variable type.
                    siz = ide->vdescr.vsize;  // Get the number of value (or of record ins array of record).
                    obj = ide->vdescr.vaddr;  // Get related address that is a pointer to the ...
                                              // ... list of field descriptor for listsy, to QuHeader for ...
                                              // ... the queuesy and dircet C/C++ variable address in all ...
                                              // ... other cases.
                    if (obj) {                // When an address is provided.
                        switch (vty) {
                            case SetUpInt:
                            case SetUpFlt:
                            case SetUpStr:
                                SetParmValue( vnam, vty, siz, obj, 0 );
                            break;
                            case SetUpDIn:
                            case SetUpDFl:
                            case SetUpDSt:
                                SetParmDynTab( vnam, vty, obj );
                            break;
                            case SetUpRec:
                                LiSetParm( vnam, (SetUpDsc*)obj );
                            break;
                            default:
                                SignalError( -16, "The SetUpMan structure for Identifier\"`\" is incorrect", vnam );
                            break;
                        }
                    } else SkipToSymbol( semicolonsy );
                } else { vnam = "<undefined>"; SkipToSymbol( semicolonsy ); }
                CheckSymbol( semicolonsy, vnam, 1 );
            } else { // if (ide) ... else ...
                SignalError( -17, " SetUpMan structure incorrect statement ", NULL );
            }
        } // while (sym.sy != eofsy) ...
    }
} // static void ParamSetting( const char * fnam, SetUpIde * idetab ).



static void ParmSave( SetUpSym ty, int siz, void *obj, int flg )
{
    EqObj          ptr;
    int col, col1, cnt;
    char          *str;

    cnt    =   0;
    col = SetUpCol;                     // Keep position to align the possible next lines.
    ptr.ad = obj;
    if (flg||siz>1) { fprintf( fp, "(" ); SetUpCol += 2; }
    while (cnt < siz) {
        switch (ty) {
            case SetUpInt: // Integer number(s).
                fprintf( fp, " %7d", ptr.ai[cnt++] ); SetUpCol += 8;
            break;
            case SetUpFlt: // floatting number(s).
                fprintf( fp, " %11.6e", ptr.af[cnt++] ); SetUpCol += 12;
            break;
            case SetUpStr: // string(s)
                str = ptr.as[cnt++];
                if (str&&str[0]) {
                    fprintf( fp, " '%s'%n", str, &col1 );
                    SetUpCol += col1;
                } else {
                    fprintf( fp, " ''" );
                    SetUpCol += 3;
                }
            break;
            default:
                ;
        }
        if (cnt<siz) {
            if (SetUpCol>=MAXLSFSZ) {
                fprintf( fp, ",\n%*s", col, " " ); SetUpCol = col;
            } else {
                fprintf( fp, ", " ); SetUpCol += 2;
            }
        }
    }
    if (flg||siz>1)    { fprintf( fp, " )" ); SetUpCol += 2; }
} // static void ParmSave( SetUpSym ty, int siz, void *obj, int flg ).



static void SavParmDynTab( SetUpSym ty, void *obj )
{
    SetUpSym       nty;
    SetUpDynTab  * src;
    int           size;
    EqObj          ptr;

    src   = (SetUpDynTab*)obj;
    size  = src->size;
    switch (ty) {
        case SetUpDIn:
            ptr.ai = src->tab.i;
            nty    =   SetUpInt;
        break;
        case SetUpDFl:
            ptr.af = src->tab.f;
            nty    = SetUpFlt;
        break;
        case SetUpDSt:
            ptr.as = src->tab.s;
            nty    = SetUpStr;
        break;
        default: ;
    }
    fprintf( fp, " %7d", size ); SetUpCol += 8;
    if (size) ParmSave( nty, size, ptr.ad, 1 );

} // static void SavParmDynTab( SetUpSym ty, void *obj ).



static void LiSySave( int siz, SetUpDsc *rec )
{
    SetUpDsc *fld = rec;
    int             col;
    
    col = SetUpCol;                     // Keep position to align the possible next lines.
    fprintf( fp, "(" ); SetUpCol ++;  
    while ((fld->vtype)!=SetUpEof) {
        ParmSave( fld->vtype, fld->vsize, fld->vaddr, 0 );
        fld++;
        if ((fld->vtype)!=SetUpEof)
            if (SetUpCol>MAXLSFSZ) { fprintf( fp, ",\n%*s", col, " " ); SetUpCol = col; }
                              else { fprintf( fp, ", " ); SetUpCol += 2; }
    }
    fprintf( fp, ")" ); SetUpCol++;
} // static void LiSySave( int siz, SetUpDsc *rec ).



static void ParamsSave( const char * fnam, const char *Appl, const SetUpIde * idetab )
// To write each identifier described in the idetab to the current setup file.
{
    const SetUpIde       * ide;
    void                 * adr;
    SetUpSym               vty;
    int                    siz;

    if (fnam&&idetab) {
        fprintf( fp, "{ SetUp file %s for %s automaticaly generated should not be edited. }\n", fnam, Appl );
        ide = idetab;                           // Get the first setup identifier.
        while ((ide->vdescr.vtype)!=SetUpEof) { // Loop on all IdeTable elements.
            vty  = ide->vdescr.vtype;
            siz  = ide->vdescr.vsize;
            adr  = ide->vdescr.vaddr;
            if (adr) {
                fprintf( fp, " %s = ", ide->idname );
                SetUpCol = strlen( ide->idname ) + 4;
                switch (vty) {
                    case SetUpInt: // integer number(s).
                    case SetUpFlt: // floatting number(s).
                    case SetUpStr: // string(s).
                        ParmSave( vty, siz, adr, 0 );
                    break;

                    case SetUpDIn:
                    case SetUpDFl:
                    case SetUpDSt:
                        SavParmDynTab( vty, adr );
                    break;

                    case SetUpRec: // Pseudo records.
                        LiSySave( siz, (SetUpDsc*)adr );
                    break;
                
                    default: ;
                }
                fprintf( fp, ";\n" );
            }
            ide++;
        }
    }
} // static void ParamsSave( const char * fnam, const char *Appl, SetUpIde * idetab ).



int  SetUpRead( const char * fpath, const SetUpIde * const idetab )
// To read a SetUp files as describe by idetab.
{
    // Try to open the setup file.
    if (!access( fpath, F_OK )) {
        if (!(fp = fopen( fpath, "r" ))) {
            SignalError( -21, "We can't open the Settup File \"`\"\n", fpath );
        } else {
            SetUpPath = fpath;
            // Now set the Draw Display Characteristic from Display_Setting files.
            ParamSetting( fpath, idetab );
            fclose( fp );
            SetUpPath = NULL;
            return 1;
        }
    }
    return 0;
} // void SetUpRead( const char * fnam, SetUpIde * idetab ).



void SetUpWrite( const char * fpath, const char * Appl, const SetUpIde * idetab )
// To create or supershed SetUp files as describe by idedtab.
{
    // Try to open the setup file.
    if (!(fp = fopen( fpath, "w" ))) {
        SignalError( -22, "We can't open the Setting File \"`\"\n", fpath );
    } else {
        SetUpPath = fpath;
        ParamsSave( fpath, Appl, idetab );
        fclose( fp );
        SetUpPath = NULL;
    }
} // void SetUpWrite( const char * fnam, const char *Appl, SetUpIde * idetab ).




//
// end of <PW-Id>.
//
