//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// Module for a View (Image and or text) for class View_GL.
//

#include <stdio.h>
#include <string.h>
#include <math.h>

#include <FL/fl_ask.H>
#include <FL/fl_draw.H>
#include <FL/Fl_Box.H>

#include "Fancy_Layout.h"
#include "Service_IO.h"
#include "Service_PrImg.h"

#include "View_GL.h"
#include "View_GL_UI.h"


extern LogFile LogF;

//#define GL_DEBUG 1

static const View_GL::Formats defrmtb[] {
  //       name        h/w (in landscape).
    { "Printer     A",  0.7071 }, // As A5, A4, A3 ...
    { "Photos  3 x 2",  0.6666 }, // ... as photo paper 15 x 10 cm ...
    { "Photos  4 x 3",  0.7500 }, // ... as 24/36 films.
//   ...
    { "Screen 16 x 9",  0.5625 }, // ... as your prefered screen X x Y.
    {        "Square",  1.0000 }, // Squared support.
//  { "Your format",    0.0    },
    0
};


Fancy_Layout            Layout; // Our layout used to render all text.



void View_SetUp::Set( int id )
{
    const View_GL::Formats * cfo = vgl->formtb_;
    char * n = 0;  // We use the void * data to transmit the Choice state.

    vfw = new Fl_Double_Window( 280, 265, "View Setup" );
        { Fl_Group* o = new Fl_Group(10, 8, 260, 75, "Format");
            o->box(FL_ENGRAVED_BOX);
            o->labelsize(20);
            o->align(Fl_Align(FL_ALIGN_TOP|FL_ALIGN_INSIDE));
                vfc = new Fl_Choice( 5, 5, 160, 25 );
                while (cfo && cfo->name) vfc->add( (cfo++)->name, 0, (Fl_Callback*)vfc_cb, (void*)(++n) );
                vfc->value( idx = id );
                vfc->callback( (Fl_Callback*)vfc_cb, (void *)this );
                vfc->when( FL_RELEASE|FL_WHEN_CHANGED );
                vfl = new Fl_Round_Button( 175,  6, 95, 20, "Landscape" );
                vfl->down_box( FL_ROUND_DOWN_BOX );
                vfl->value( vgl->ratiof_ >= 1.0 ? 0 : 1 );
                vfp = new Fl_Round_Button( 175, 25, 95, 20, "Portrait" );
                vfp->down_box( FL_ROUND_DOWN_BOX );
                vfp->value( vgl->ratiof_ >= 1.0 ? 1 : 0 );
                vfl->callback( (Fl_Callback*)vfl_cb, (void *)vfp );
                vfp->callback( (Fl_Callback*)vfp_cb, (void *)vfl );
            o->end();
        } // Fl_Group* o
        { Fl_Group* o = new Fl_Group(10, 90, 260, 95, "Preferred layout");
            o->box(FL_ENGRAVED_BOX);
            o->labelsize(20);
            o->align(Fl_Align(FL_ALIGN_TOP|FL_ALIGN_INSIDE));
                glf = new Fl_Round_Button(30, 120, 100, 20, "Left");
                glf->down_box(FL_ROUND_DOWN_BOX);
                glf->value( vgl->geomet_&View_GL::Img_Right ? 0 : 1 );
                gri = new Fl_Round_Button(30, 145, 100, 20, "Right");
                gri->down_box(FL_ROUND_DOWN_BOX);
                gri->value( vgl->geomet_&View_GL::Img_Right ? 1 : 0 );
                glf->callback((Fl_Callback*)glf_cb, (void *) gri );
                gri->callback((Fl_Callback*)gri_cb, (void *) glf );
                gto = new Fl_Round_Button(160, 120, 100, 20, "Top");
                gto->down_box(FL_ROUND_DOWN_BOX);
                gto->value( vgl->geomet_&View_GL::Img_Top ? 1 : 0 );
                gbo = new Fl_Round_Button(160, 145, 95, 20, "Bottom");
                gbo->down_box(FL_ROUND_DOWN_BOX);
                gbo->value( vgl->geomet_&View_GL::Img_Top ? 0 : 1 );
                gto->callback((Fl_Callback*)gto_cb, (void *) gbo );
                gbo->callback((Fl_Callback*)gbo_cb, (void *) gto );
            o->end();
        } // Fl_Group* o
        gma = new Fl_Value_Input(225, 195, 42, 20, "Text Margin");
        gma->labelsize(20);
        gma->value( vgl->margin_ );
        gma->textfont( FL_HELVETICA );
        gma->textsize(14);
        { Fl_Button* o = new Fl_Button(10, 230, 95, 25, "Quit");
            o->color((Fl_Color)62);
            o->labelsize(20);
            o->callback((Fl_Callback*)vfw_cb, (void *)this );
        }
        vfr = new Fl_Return_Button(175, 230, 95, 25, "Select");
        vfr->labelsize(24);
        vfr->labelfont( FL_HELVETICA_BOLD );
        vfr->callback( (Fl_Callback*)vfr_cb, (void *)vgl );
    vfw->end();
    vfw->box(FL_BORDER_BOX);
    vfw->labelfont( FL_HELVETICA_BOLD );
    vfw->labelsize(16);
    vfw->callback( (Fl_Callback*)vfw_cb, (void *)this );
//  printf( " %d, %s\n", vgl->largc_, vgl->largv_[0] ); fflush( stdout );
    vfw->show( vgl->largc_, vgl->largv_ );
} // void View_SetUp::Set().





View_GL::View_GL( int x, int y, int w, int h, const char * L ):
          Fl_Gl_Window( x, y, w, h, L )
// Here we are called by the creator View_GL_UI(). After ...
// ... the Wiew_GL_UI creator will init Vgl_ui.
{
    i_map_ = 0; i_text_ = 0;
} // View_GL::View( int x, int y, int w, int h, const char * L ).




void View_GL::Init()
//
// To initialize the View_GL Module.
//
{
    vfrm_      = 0;                     // Initialize the View_Format system.

    W_win      = Vgl_ui->w_win;         // Set all View_GL_UI links : The main window,
    W_view     = Vgl_ui->w_view;        //      the sub group of image to display,
    W_frame    = Vgl_ui->w_frame;       //      the frame Menu_Item
    W_ftname   = Vgl_ui->w_ftname;      //      the font name display/setup, and
    W_ftsize   = Vgl_ui->w_ftsize;      //      related font size,
    W_xscroll  = Vgl_ui->w_xscroll;     //      and X and Y scroll bars.
    W_yscroll  = Vgl_ui->w_yscroll;

//  printf( " Call 1 of Analyse\n"); fflush( stdout );
//  Analyse();

    Fw_win     =                     0; //

    formtb_    =               defrmtb; // Install the default format table.
    ratiof_    =      formtb_[0].ratio; // Default to A4 Landscape.
    margin_    =                    15; // Margin to save between text and image.
//  txtprp_    =                  0.25; // Part of surface for the text.
    i_text_    =                     0; // No text at the creatimaxword_advon of View.


#ifdef WIN32
    const char * fntpath = "C:\\Users\\pierr\\AppData\\Local\\.DiaViewer\\Dancing_Script_OT.otf";
#else
    const char * fntpath = "/usr/local/share/fonts/Dancing_Script_OT.otf"; // Ne peut pas fonctionner sur window.
#endif

    usdfnt_    = new FTPolygonFont( fntpath ); // It is the best for our use.
//  usdfnt_    = new FTTextureFont( fntpath );

    if (usdfnt_->Error()) {
        LogF.Write( " Cannot use the GL font \"%s\".\n", fntpath );
        fl_alert( "View_GL: Cannot open the font \"%s\".", fntpath );
        exit( 1 );
    }

    usdfnt_->CharMap( ft_encoding_unicode );
    if (!usdfnt_->FaceSize(64)) {
        LogF.Write( " View_GL: Failed to set size %d for GL font \"%s\".\n", 64, fntpath );
        fl_alert( "View_GL: Cannot set size %d for font \"%s\".", 64, fntpath );
        exit( 1 );
    }

    Layout.SetFont( usdfnt_ );
    Layout.SetLinesLimits( BLine, ELine, (void*) this );

} // void View_GL::Init().




View_GL * View_GL::Create( int w, int h, const char * L )
{
    if (w < 600) w = 600;               // Set for a window just or above the minimum sizes.
    if (h < 400) h = 400;
    View_GL_UI * vui = new View_GL_UI( 0, 0, w, h, L ); // Create View_GL_UI with inside View_GL.
    vui->w_view->Vgl_ui = vui;                          // Establish the link of View_GL and View_GL_UI.
    vui->w_view->Init();
//  printf( " Call 2 of Analyse\n"); fflush( stdout );
//  vui->w_view->Analyse();
    return vui->w_view;
} // View_GL * View_GL::Create().



void View_GL::Set_Conf( float f, int g, int mg, int fs, char * fn )
{
    int                    ii = 0;
    const Formats * lst = formtb_;
//printf( " Old r = %f\n", f );
    while (lst && lst->name) {
        if (abs( lst->ratio - f ) < 1.0E-4) { ratiof_ = lst->ratio; break; }
        if (abs( lst->ratio - 1.0/f) < 1.0E-4) { ratiof_ = 1.0/lst->ratio; break; }
        lst++; ii++;
    }
    if (lst&&lst->name) frmidx_ = ii;
    else { ratiof_ = formtb_[0].ratio; frmidx_ = 0; }

//printf( " Ratio = %f, idx = %d\n", ratiof_, frmidx_ );
    geomet_ = (ImgGeo)g; margin_ = mg; vfsize_ = fs; vfname_ = fn;
    if (!margin_) margin_ = 15;

} // View_GL::Set_Conf( float f, int g, int mg, int fs, char * fn ).



char * View_GL::Text_Parsing( const char * txt )
{
    char     buf[4094];
    int   bf,  ii,  dw;
    unsigned char   ch;
    const char    * cb;
    GLdouble   wl,  ml;

    cb  = txt;
    if (cb[0] == '*' && cb[1] == '[') { // We skip any rankind information.
        cb += 2;
        while (*cb && *cb != ']') cb++;
        cb++;
        if (*cb == '.') cb++;
    }

    // Evaluate the surface to give for the text.

    ml = 0;
    ii = 0; dw = -1;
    ch = ' ';
    bf =   1; // To flag the first word.
    while (ch) {
        // Skip any space except TAB if the next char is > ' ' :
        if (ch <= ' ' && !(ch == '\t' && *cb > ' '))
            while (ch = *(cb++))
                if (ch > ' ' || (ch == '\t' && *cb > ' ')) break;
        if (ch == '\t') { buf[ii++] = '\n'; buf[ii++] = ch; ch = *(cb++); }
        if (ch) {                       // When some word(s) are specified except for ...
            if (!bf) buf[ii++] = ' ';   // Insert one space between each word (but not for the first one).
               else bf = 0;
            dw = ii; buf[ii++] = ch;
            while ((ch = *(cb++)) > ' ') buf[ii++] = ch;
            // The Advance for a word is in font units (not scaled to GL unit here).
            wl = d_tsc_*usdfnt_->Advance( buf+dw, ii - dw );
            if (wl < ml) ml = wl;
            dw = -1;                    // A new word is append in the word table.
        }
    }
    buf[ii] = 0;

    d_mxw_ = ml; // Set the length of the longest word.
    return strdup( buf );
} // void View_GL::Text_Parsing().



void View_GL::Slide( const char * txt, Image_REF * img,
                     const char * Vn, int ifl, int inb )
{
    if (img) {
        i_pcs_ = img->P();              // Get the image sizes,
        i_shw_ = i_hwi_ = (i_wid_ = img->W())/2;
        i_shh_ = i_hhe_ = (i_hei_ = img->H())/2;
        i_cex_ = i_cey_ = 0;
        i_mx1_ = -(i_mx2_ = i_hwi_);
        i_my1_ = -(i_my2_ = i_hhe_);
        i_map_ = img->Map();
    } else i_map_ = 0;                  // No image.
//  if (i_text_) delete[] i_text_;      // Get the related text if specified.
    if (txt && txt[0]) i_text_ = Text_Parsing( txt );
                  else i_text_ =   0;    // No text.
    snprintf( wlabel_, 127, "%s: %3d/%3d", Vn, ifl, inb );
    W_win->label( wlabel_ );

//  FirstView = 1;
//  printf( "Slide called\n");
} // void View_GL::Slide( const char * txt, Image_Ref img ).



//
// Back Call functions :
//

// Called when we quit the View environment.
void View_GL::Quit_CB() {
    evfunc_( 0, evdata_ );
    delete W_win;
    W_win = 0;
} // void View_GL::Quit_CB( Fl_Widget *, void * data ).



// Called when on a View_GL_UI Format setting request.
void View_GL::FrmGeo_CB()
{
    if (!vfrm_) vfrm_ = new View_SetUp( this );
//  vfrm_->CallBack( ChangeFormat_CB );
    vfrm_->Set( frmidx_ );
} // void View_GL::FrmGeo_CB().



// Directly called by View_SetUp::vfr_cb() (callback of Select bouton.)
void View_GL::SetFrmGeo()
{
//  View_SetUp * v = Vgl_ui;

    frmidx_ = vfrm_->idx;
    if (frmidx_ >= 0) {
        float r = formtb_[frmidx_].ratio;
        if (r > 0.1)ratiof_ = (vfrm_->vfl->value()) ? r : 1.0/r;
               else ratiof_ = 1.0;
    }
    geomet_ = vfrm_->gto->value() ? Img_Top : Img_Undef;
    if (vfrm_->gbo->value()) geomet_ |= Img_Bottom;
    if (vfrm_->glf->value()) geomet_ |= Img_Left;
    if (vfrm_->gri->value()) geomet_ |= Img_Right;
    margin_ = vfrm_->gma->value();

    redraw();
} // void View_GL::SetFrmGeo().


/*
// Called by cfr_cb to set the requested format change.
void View_GL::ChangeFormat_CB( int idf, int laf, View_GL * v )
{
//printf( " Change format %d %d\n", laf, idf );
    v->frmidx_ = idf;
    // Get the Format ratio ...
    if (idf >= 0) {
        float r =  v->formtb_[idf].ratio;
        if (r > 0.1) v->ratiof_ = laf ? r : 1.0/r;
                else v->ratiof_ = 1.0;
//printf( " idf = %d, laf = %d, R = %f\n", idf, laf, v->ratiof_ );
    }
    // and set the OpenGL output surface.
    v->redraw();
} // void View_GL::ChangeFormat_CB( int laf, int idf ).
*/



/*
void View_GL::Format_CB()
{
    Fw_Viewfrm = new ViewFrm( ( formtb_ );
} // void View_GL::Format_CB().


void View_GL::Geometry_CB() {

} // void View_GL::Geometry_CB( Fl_Widget *, void * data ).
*/


void View_GL::Print_CB() {
//  if () {
        Smp_Printer::Print_Widget( W_view, 2.0*d_hwi_, 2.0*d_hhe_ );
//  }

} // void View_GL::Print_CB( Fl_Widget *, void * data ).


void View_GL::SaveAs_CB() {

} // void View_GL::SaveAs_CB( Fl_Widget *, void * data ).


void View_GL::UnZoom_CB( int qf ) {
    i_shw_ = i_hwi_; i_shh_ = i_hhe_;
    i_cex_ = i_cey_ = 0;
    redraw();
} // void View_GL::UnZoom_CB( Fl_Widget *, void * data ).


void View_GL::Frame_CB() {

} // void View_GL::Frame_CB( Fl_Widget *, void * data ).


void View_GL::About_CB() {

} // void View_GL::About_CB( Fl_Widget *, void * data ).


void View_GL::Help_CB() {

} // void View_GL::Help_CB_CB( Fl_Widget *, void * data ).


void View_GL::ChangeFont_CB() {

} // void View_GL::ChangeFont_CB_CB( Fl_Widget *, void * data ).


void View_GL::ChangeFtsize_CB() {

} // void View_GL::ChangeFtsize_CB( Fl_Widget *, void * data ).




void View_GL::YScroll_CB()
{
  GLint vl, sz;

  sz = i_hei_ - i_nrow;
  vl = sz - W_yscroll->value();
  if (vl >= 0 && vl < sz) {
//      printf( " X Scroll val = %d\n", vl  );
    Y_shift += vl - i_srow;
    redraw();
  }
} // void View_GL::YScroll_CB().



void View_GL::XScroll_CB()
{
  GLint vl;

  vl = W_xscroll->value();
  if (vl >= 0 && vl < i_wid_ - i_npix) {
//      printf( " X Scroll val = %d\n", vl  );
    X_shift += vl - i_spix;
    redraw();
  }
} // void View_GL::XScroll_CB().



//
// Private methods.
//



void View_GL::XYinImage( int &x, int &y )
// Get the pointed coordinate in screen pixels space (origine at center of View_GL window).
{ // event_x, event_y give the coordinates relative to the Gl_Window (origine at the center).
  x = Fl::event_x() - w()/2;
  y = h()/2 - Fl::event_y();
} // void View_GL::XYinImage( int &x, int &y ).



void View_GL::MouseDown( int x, int y, int button )
{
  if (!MouseStat) {     // When the mouse is active without button already pushed.
    w_zx2_ = w_zx1_ = x, w_zy2_ = w_zy1_ = y;
    fl_cursor( FL_CURSOR_CROSS );
    UsedButton = button;
    MouseStat = 1;
  }
} // void View_GL::MouseDown( int x, int y, int button ).



void View_GL::MouseDrag( int x, int y, int button )
{
  if (UsedButton!=button) {
    MouseStat = 0; return;
  }
  if ((x!=w_zx2_)||(y!=w_zy2_)) {
    w_zx2_ = x; w_zy2_ = y;
    redraw();
  }
} // void View_GL::MouseDrag( int x, int y, int button ).



void View_GL::MouseUp( int x, int y, int button )
{
    GLdouble zw, zh;

    if (UsedButton != button) { MouseStat = 0; return; }
    w_zx2_ = x; w_zy2_ = y;
    fl_cursor( FL_CURSOR_DEFAULT );
    MouseStat = 0;

    d_zhw_ = abs( w_zx2_ - w_zx1_ )/2.0;        // Compute the size of zoom rectangle ...
    d_zhh_ = abs( w_zy2_ - w_zy1_ )/2.0;
    d_zcx_ = (w_zx1_ + w_zx2_)/2.0;             // ... and zoom rectangle center.
    d_zcy_ = (w_zy1_ + w_zy2_)/2.0;

//printf( " ini Z rect x1=%f, y1=%f, x2=%f, y2=%f\n    mpx=%f, mpy=%f\n",
//        d_zcx_-d_zhw_, d_zcy_-d_zhh_, d_zcx_+d_zhw_, d_zcy_+d_zhh_, di_mpx_, di_mpy_ );


    if (d_zhw_ < di_hwi_ && d_zhh_ < di_hhe_ &&         // We reject any too large or ...
        abs( d_zcx_ )<di_hwi_ && abs( d_zcy_ )<di_hhe_) // ... out of range zoom rectangle.
    {
        // We shift any rectangles that straddle the boundaries.
        if (d_zcx_ - d_zhw_ < -di_hwi_) d_zcx_ = d_zhw_ - d_hwi_;
        else if (d_zcx_ + d_zhw_ > di_hwi_) d_zcx_ = di_hwi_ - d_zhw_;
        if (d_zcy_ - d_zhh_ < -di_hhe_) d_zcy_ = d_zhh_ - d_hhe_;
        else if (d_zcy_ + d_zhh_ > di_hhe_) d_zcy_ = di_hhe_ - d_zhh_;
//printf( " w shift Z rect x1=%f, y1=%f, x2=%f, y2=%f\n    mpx=%f, mpy=%f\n",
//        d_zcx_-d_zhw_, d_zcy_-d_zhh_, d_zcx_+d_zhw_, d_zcy_+d_zhh_, di_mpx_, di_mpy_ );

        // Now we must change the heigth/width ratio to fit with the Format.
        if (di_hhe_*d_zhw_ > d_zhh_*di_hwi_) d_zhh_ = di_hhe_*d_zhw_/di_hwi_;
                                        else d_zhw_ = d_zhh_*di_hwi_/di_hhe_;
printf( " fini Z rect x1=%f, y1=%f, x2=%f, y2=%f\n    mpx=%f, mpy=%f\n",
        d_zcx_-d_zhw_, d_zcy_-d_zhh_, d_zcx_+d_zhw_, d_zcy_+d_zhh_, di_mpx_, di_mpy_ );

        // Now get the Zoom rectangle cord. and sizes in the image space.
        i_zcx_ = round( d_zcx_/r_scale_ );
        i_zcy_ = round( d_zcy_/r_scale_ );
        i_zsw_ = round( d_zhw_/r_scale_ ); i_zsh_ = round( d_zhh_/r_scale_);

        // We reject any zoom that would give a resolution better than that of the photo.
        if (di_hwi_/i_zsw_ < 2.0 ) ZoomStatus = 1;

#ifdef GL_DEBUG
        printf( " SCALE = %f\n", r_scale_ );
        printf( " Z rect (d coord): x1=%d, y1=%d, x2=%d, y2=%d\n", w_zx1_, w_zy1_, w_zx2_, w_zy2_ );
        printf( " Z rect (i coord): x1=%d, y1=%d, x2=%d, y2=%d\n", i_zcx_-i_zsw_, i_zcy_-i_zsh_,
                                                                   i_zcx_+i_zsw_, i_zcy_+i_zsh_ );
        printf( " Center at (%d, %d)\n", i_zcx_, i_zcy_ );
#endif
        redraw();
    }
} // void View_GL::MouseUp( int x, int y, int button ).



int  View_GL::handle( int evn )
{
    int       mx,   my;
    int       ky,   bt;

//  printf( " Handle!\n" ); fflush( stdout );

      bt = Fl::event_button();
  switch (evn) {
    case FL_LEAVE: // The mouse has moved out of the DiaVIewer_GL widget.
//    printf( " Go out of GL Widget\n" );
      clear_visible_focus();

      return 1;
    break;

    case FL_ENTER:
//    printf( " Enter in the GL Widget\n" );
      return 1;
    break;

    case FL_FOCUS:
    case FL_UNFOCUS:
      return 1;
    break;

    case FL_PUSH:  //
      if (Fl::event_inside(this)) set_visible_focus();
                             else clear_visible_focus();
//    printf( " Focus is %d\n", Fl::visible_focus() );
      if (Fl::visible_focus())take_focus();
      XYinImage( mx, my );
      MouseDown( mx, my, bt );
      return 1;
    break;

    case FL_DRAG:
      XYinImage( mx, my );
      MouseDrag( mx, my, bt );
      return 1; redraw();
    break;

    case FL_RELEASE:
      XYinImage( mx, my );
      MouseUp( mx, my, bt );
      return 1;
    break;


//  case FL_MOUSEWHEEL:
//
//  break;

    case FL_KEYBOARD:
      switch (ky = Fl::event_key()) {
        case FL_Left:
          if (Fl::event_state( FL_SHIFT )) evfunc_( -2, evdata_ );
                                      else evfunc_( -1, evdata_ );
          return 1;
        break;

        case FL_Right:
          if (Fl::event_state( FL_SHIFT )) evfunc_(  2, evdata_ );
                                      else evfunc_(  1, evdata_ );
          return 1;
        break;

        case FL_Up:

        break;

        case FL_Down:

        break;

        default: ;
      }
    default: ;
  }
  return Fl_Gl_Window::handle( evn );


} // int  View_GL::handle( int ev ).





void View_GL::Set_Format()
//
// Compute the size and limits of the OpenGL ViewPort in the actual View_GL window ...
// ... to can plot the frame (outline) of the View.
//
// Must be called by draw().
//
{
    int  w_vpx,  w_vpy, // View Group (up,left) origine coordinates,
         w_vpw,  w_vph, // View Group sizes (width anf height),
         vp_cx,  vp_cy; // View Group center coordinates.
    double          rg;

    w_vpx = W_view->x(); w_vpy = W_view->y();   // Get View Group origine and sizes.
    w_vpw = W_view->w(); w_vph = W_view->h();
    vp_cx = w_vpw/2 + w_vpx;                    // Get the View Center coordinates.
    vp_cy = w_vph/2 + w_vpy;

    rg = double( w_vph )/double( w_vpw );       // Compute the H/W ratio.

    // The GL Viewport must be have the H/W ratio egual to ratiof_.

    if (rg < ratiof_) {                         // If window H/W < Format Ratio ...
        w_glh_ = w_vph; w_glw_ = w_vph/ratiof_; // ... we use the full height for the GL space and ...
        w_glx_ = (w_vpw - w_glw_)/2;            // ... set the Left side of frame at the middle ...
        w_gly_ = 0;                             // ... of View_GL group (for GL y=0 at bottom).,
    } else {                                    // ... else ...
        w_glw_ = w_vpw; w_glh_ = w_vpw*ratiof_; // ... we use the full width and set the GL space at ...
        w_gly_ = (w_vph - w_glh_)/2;            // ... the center of View_GL group Space.
        w_glx_ = 0;
    }
    w_ghw_ = w_glw_/2; w_ghh_ = w_glh_/2;       // Set the half of GL window sizes (in screen pixels).

    // Build the document space limits.
    di_hwi_ = d_hwi_ = double( w_ghw_ );        // Define the document size in document pixels.
    di_hhe_ = d_hhe_ = double( w_ghh_ );
    di_cex_ = di_cey_ = 0.0;                    // Set origine at ViewPort (GL Space) center.

    // Here, the w_glx and w_gly are the System Window coordinates (with origin at the Up Left corner),
    // but for GL the origine is in the bottom left corner in screen reference (glViewport)
    // Our glOrtho space the origine is at the center of a space with the sizes in specified format
    // proportions (ratiof_).
    //
    // For the default géometry (geomet_ == Img_Undef), the image is the format specified
    // space (Image H/F is at the center of View_Gl space witn the maximum used surface.
    // No specific text place is allocated.

} // View_GL::Set_Format( float R ).




GLdouble View_GL::TextSurface()
{ // Get the surface for a text.
    FTBBox b;

    b = usdfnt_->BBox( i_text_ );
    return d_tsc_*d_tsc_*(b.Upper().Xf() - b.Lower().Xf())*usdfnt_->LineHeight();
} // void View_GL::TextSurface( const char * txt ).


GLdouble View_GL::HMeasure()
{
    FTBBox b = Layout.BBox( i_text_ );  // The functions BLine and ELine set the lines parameters.

    return d_tsc_*(b.Upper().Yf() - b.Lower().Yf()); // Just, we return a value in GL coordinates.
} // GLdouble View_GL::HMeasure().



void View_GL::Set_Geometry()
//
// Compute the space for image and text.
//
{
    const GLdouble Lred = 0.80,         // We lose 20% of surface for text, Too short lines in L configuration.
                   Chsz = 0.04,         // Choice for Character size relative to format (4%).
                   Lsmi = 10.0;         // The minimum length of one text line (relative to line height).

    // We add 10% of surface for text, for L configuration.

    if (geomet_ == Img_Undef) geomet_ = Img_Top|Img_Right;

    geom_ =  Img_Undef;
    int       fv,   nw;

    GLdouble     rw,   rh,   re,   tmpl, // Temporary image scales, sizes ...
           di_imsrf, di_txsrf, di_shsrf, // ... and surfaces.
           di_mLsrf,  tmp_imsrf,  twmax; // Maximum for Text column width.

    if (!(i_text_ && i_text_[0])) {      // When no text exist we set the central image display mode.
        di_hwi_ = d_hwi_; di_hhe_ = d_hhe_;
        di_cex_ = di_cey_ = 0.0;
        di_mpx_ = di_cex_ - di_hwi_;
        di_mpy_ = di_cey_ - di_hhe_;
        return;
    }

    Layout.SetAlignment(FTGL::ALIGN_JUSTIFY );

    d_tss_ = w_glh_*Chsz;               // Define our character sizes.
    d_tlh_ = usdfnt_->LineHeight();     // To evaluate the area size for the text (vertical space for one line).
    d_tla_ = usdfnt_->Ascender();       // Shift to keep from the top of text layout.
    d_tsc_ = d_tss_/d_tlh_;             // Compute the text relative scale.
    d_tlh_ *= d_tsc_; d_tla_ *= d_tsc_; // d_tlh_ and dtla_ put in glOrtho units.

    d_lmi_ = d_tlh_*Lsmi;               // Set the minimum layout width for lines of text.

    // Estimate the area required for our text.

    di_txsrf = TextSurface();           // Compute the text surface.
    di_shsrf = 4.0 * d_hwi_ * d_hhe_;   // The total (sheet or screen surface)/4.


// Step 1 : Find the maximum surface for image with the image ratio H/W on the total surface.
    rw = d_hwi_ / i_hwi_; rh = d_hhe_ / i_hhe_;
    if (rw < rh) {
        fv = 0; r_scale_ = rw;                  // Flag for Horizontal mode (H).
        di_hwi_ = d_hwi_; di_hhe_ = rw*i_hhe_;  // We get the image sizes above/below the text.
        tmp_imsrf = 4.0 * di_hwi_ * di_hhe_;    // Keep the reference image surface for "H" config.
        d_tw1_ = 2.0*(d_hwi_ - margin_);        // The Text zone has the maximal width.

        d_th1_ = HMeasure();
//      d_th1_ = TextSizes( 0, d_tw1_, txtbuf );
        // Now we know the exact size of the text area in the "H" config.
        if (d_th1_ > 2.0*(d_hhe_ - di_hhe_ - margin_))
        { // The free height is too small, we must reduce the image size.
            tmpl = d_hhe_ -margin_ - d_th1_*0.5;// Is the new height for the image in "H" config.
            if (tmpl / di_hhe_ > 0.70) {        // If this reduction is less than 30% we can accept ...
                di_hwi_ *= tmpl / di_hhe_;      // ~ to di_hwi = tmpl * di_hwi_ / di_hhe_;
                di_hhe_ = tmpl;
                geom_ = Img_Top|Img_Bottom;     // The prefered choice (text above or below of image).
            }                                   // we should use the "L" configuration.
        } else geom_ = Img_Top|Img_Bottom;      // The prefered choice (text above or below of image).
    } else {
        fv = 1;                                 // Flag for Vertical mode (V).
        r_scale_ = rh;
        di_hhe_ = d_hhe_; di_hwi_ = rh*i_hwi_;  // We get the image sizes at left or right of text.
        d_tw1_ = 2.0*(d_hwi_-di_hwi_ -margin_); // Is the remaining width for the text.
        tmp_imsrf = 4.0 * di_hwi_ * di_hhe_;    // Keep the reference image surface for "V" config.
        if (d_tw1_ > d_lmi_) {                  // if this text column is not too narrow ...
            if (HMeasure() <= 2.0*(d_hhe_-margin_))
//          d_th1_ = TextSizes( 0, d_tw1_,txtbuf );
//          if (d_th1_ <= 2.0*(d_hhe_-margin_)) // ... and if its heigth is enough large ...
                geom_ = Img_Left|Img_Right;     // The prefered choice (text at left or right of image).
        } else {                                // We try to reduce the image.
            d_tw1_ = d_lmi_;                    // We force the free colomn to be enough large.
            d_th1_ = 2.0*(d_hhe_ - margin_);    // We want this maximum for text heigth.
            twmax = d_hwi_*0.30;
            do {
                if (HMeasure() <= d_th1_) break;
//              if (TextSizes( 0, d_tw1_, txtbuf ) <= d_th1_) break;
                d_tw1_ += 0.05*d_hwi_;
            } while (d_tw1_ <= twmax);
            if (d_tw1_ <= twmax) {
                tmpl = d_hwi_ - margin_ - 0.5*d_tw1_;
                di_hhe_ *= tmpl / di_hhe_;      // ~ di_hhe_ = tmpl * di_hwi / di_hhe_;
                di_hwi_ = tmpl;
                // Here, it is possible that an L-shaped configuration would be more advantageous.

                geom_ = Img_Left|Img_Right;
            }
        }
    }


    if (geom_ == Img_Undef) {                   // We use the "L" configuration.
        fv = -1;
        // In the "L" configuration the margin surface is a constante (depend only of the FOrmat and margin_).
        di_mLsrf = 4.0*margin_*(d_hwi_ + d_hhe_ - margin_); // for "L" text area Configuration,
        di_imsrf = (di_shsrf - di_mLsrf - di_txsrf)*Lred;
        // This surface is reserved for the image.
        re = sqrt( di_imsrf / tmp_imsrf );      // We compute the variation sizes factor ...
        if (re < 1.0) {
            di_hwi_ *= re; di_hhe_ *= re;       // and apply it.
        }

        // Exchange part of text area must  present an integer number of Text lines.
        // This is especially important so that the high and low areas of the "L"
        // configuration have an invisible transition zone.
        //

        geom_ = Img_Right|Img_Top | Img_Left|Img_Top | Img_Left|Img_Bottom | Img_Bottom|Img_Right;
    }

//printf( " geometry offre = %d, souhaits = %d\n", geom_, geomet_ );

    if (geom_ == Img_Vertical) geom_ &= geomet_;
    else
    if (geom_ == Img_Horizontal) geom_ &= geomet_;
    else geom_ = geomet_;
//printf( " Géometrie Adoptée %d\n", geom_ );


    d_tw2_ = d_th2_ = 0.0;              // Not used for centered geometry.
    // The text origine design the top left corner of layouts.
    switch (geom_){
        case Img_Top: { // printf( " TOP\n" );
            di_cex_ = 0.0;
            di_cey_ = d_hhe_ - di_hhe_;
            d_tw1_  = 2.0*(d_hwi_ - margin_);
            d_th1_  = 2.0*(d_hhe_ - di_hhe_ - margin_);
            d_tx1_  = margin_ - d_hwi_;
            d_ty1_  = di_cey_ - di_hhe_ - margin_;
            break;
        }
        case Img_Bottom: { // printf( " BOTTOM\n" );
            di_cex_ = 0.0;
            di_cey_ = di_hhe_ - d_hhe_;
            d_tw1_  = 2.0*(d_hwi_ - margin_);
            d_th1_  = 2.0*(d_hhe_ - di_hhe_ - margin_);
            d_tx1_  = margin_ - d_hwi_;
            d_ty1_  = d_hhe_ - margin_;
            break;
        }
        case Img_Right: { // printf( " RIGHT\n" );
            di_cex_ = d_hwi_ - di_hwi_;
            d_tw1_  = 2.0*(d_hwi_ - di_hwi_ - margin_);
            d_th1_  = 2.0*(d_hhe_ - margin_);
            d_tx1_  = margin_ - d_hwi_;
            d_ty1_  = d_hhe_ - margin_;
            break;
        }
        case Img_Left: { // printf( " LEFT\n" );
            di_cex_ = di_hwi_ - d_hwi_;
            d_tw1_  = 2.0*(d_hwi_ - di_hwi_ - margin_);
            d_th1_  = 2.0*(d_hhe_ - margin_);
            d_tx1_  = di_cex_ + di_hwi_ + margin_;
            d_ty1_  = d_hhe_ - margin_;
            break;
        }
        case Img_Top|Img_Right: { // printf( " UP RIGHT\n" );
            di_cex_ = d_hwi_ - di_hwi_;
            di_cey_ = d_hhe_ - di_hhe_;
            d_tw1_  = 2.0*(d_hwi_ - di_hwi_ - margin_);
            d_th1_  = 2.0*di_hhe_;
            d_tx1_  = margin_ - d_hwi_;
            d_ty1_  = d_hhe_ - margin_;
            d_tw2_  = 2.0*(d_hwi_ - margin_);
            d_th2_  = 2.0*(d_hhe_ - di_hhe_ - margin_);
            d_tx2_  = d_tx1_;
            d_ty2_  = d_ty1_ - d_th1_;
            break;
        }
        case Img_Top|Img_Left: { // printf( " UP LEFT\n" );
            di_cex_ = di_hwi_ - d_hwi_;
            di_cey_ = d_hhe_ - di_hhe_;
            d_tw1_  = 2.0*(d_hwi_ - di_hwi_ - margin_);
            d_th1_  = 2.0*di_hhe_;
            d_tx1_  = d_hwi_ - d_tw1_ - margin_;
            d_ty1_  = d_hhe_ - margin_;
            d_tw2_  = 2.0*(d_hwi_ - margin_);
            d_th2_  = 2.0*(d_hhe_ - di_hhe_ - margin_);
            d_tx2_  = margin_ - d_hwi_;
            d_ty2_  = d_ty1_ - d_th1_;
            break;
        }
        case Img_Bottom|Img_Left: { // printf( " BOTTOM LEFT\n" );
            di_cex_ = di_hwi_ - d_hwi_;
            di_cey_ = di_hhe_ - d_hhe_;
            d_tw1_  = 2.0*(d_hwi_ - margin_);

            d_th1_  = 2.0*(d_hhe_ - di_hhe_ - margin_);
            d_tx1_  = margin_ - d_hwi_;
            d_ty1_  = d_hhe_ - margin_;
            d_tw2_  = 2.0*(d_hwi_ - di_hwi_ - margin_);
            d_th2_  = 2.0*di_hhe_;
            d_tx2_  = di_cex_ + di_hwi_ + margin_;
            d_ty2_  = d_ty1_ - d_th1_;
            break;
        }
        case Img_Bottom|Img_Right: { // printf( " BOTTOM RIGHT\n" );
            di_cex_ = d_hwi_ - di_hwi_;
            di_cey_ = di_hhe_ - d_hhe_;
            d_tw1_  = 2.0*(d_hwi_ - margin_);
            d_th1_  = 2.0*(d_hhe_ - di_hhe_ - margin_);
            d_tx1_  = margin_ - d_hwi_;
            d_ty1_  = d_hhe_ - margin_;
            d_tw2_  = 2.0*(d_hwi_ - di_hwi_ - margin_);
            d_th2_  = 2.0*di_hhe_;
            d_tx2_  = d_hwi_ - margin_;
            d_ty2_  = d_ty1_ - d_th1_;

            break;
        }
    }
    di_mpx_ = di_cex_ - di_hwi_;
    di_mpy_ = di_cey_ - di_hhe_;


#ifdef GL_DEBUG

    printf( " Image origine GL coord  : %f, %f\n", di_mpx_, di_mpy_ );
    printf( " Image Right, top at %f, %f\n", di_cex_ + di_hwi_, di_cey_ - di_hhe_ );


#endif

} // void View_GL::Set_Geometry().


float View_GL::BLine_i( float y )
{
   switch (geom_) {
        case Img_Top|Img_Left:  { // printf( " UP LEFT\n" );
            if (y > -d_th1_/d_tsc_) return 0.0;
                        else return (d_tw1_ - d_tw2_)/d_tsc_;
        }
        case Img_Bottom|Img_Left: { // printf( " BOTTOM LEFT\n" );
            if (y > -d_th1_/d_tsc_) return 0.0;
                        else return (d_tw1_ - d_tw2_)/d_tsc_;
        }
//      case Img_Bottom|Img_Right: { // printf( " BOTTOM RIGHT\n" );
//          return 0.0;
//      }
        default:
            return 0.0;
   }
} // float View_GL::BLine_i( float y ).



// For these two funtions .
// The Origine is d_tx1, d_ty1_ in config H opr V.
// For L config, The origine in the UP-Left cornor of the rectangle with the two text area.
float View_GL::BLine( float y, void * d )
{
    return ((View_GL*)d)->BLine_i( y );
} // float View_GL::BLine( float y, void * d ).



float View_GL::ELine_i( float y )
{
    switch (geom_) {
        case Img_Top|Img_Right:  //  { // printf( " UP RIGHT\n" );
            if (y > - d_th1_/d_tsc_) return d_tw1_/d_tsc_;
                                else return d_tw2_/d_tsc_;
//      }
        case Img_Top|Img_Left:     { // printf( " UP LEFT\n" );
            return d_tw1_/d_tsc_;
        }
        case Img_Bottom|Img_Left:  { // printf( " BOTTOM LEFT\n" );
            return d_tw1_/d_tsc_;
        }
        case Img_Bottom|Img_Right: { // printf( " BOTTOM RIGHT\n" );
            if (y > - d_th1_/d_tsc_) return d_tw1_/d_tsc_;
                                else return d_tw2_/d_tsc_;
        }
        default:
            return d_tw1_/d_tsc_;
    }
} // float View_GL::ELine_i( float y ).


float View_GL::ELine( float y, void * d )
{
    return ((View_GL*)d)->ELine_i( y );
} // float View_GL::ELine( float y, void * d  ).





void View_GL::Write_Text()
{
    GLdouble    tm1x, tm1y, tm2x, tm2y;

//  FTSimpleLayout      zone_1;
//  FTBBox              tbox_1;
/*
    zone_1.SetLineLength( d_tw1_/d_tsc_ );
    zone_1.SetFont( usdfnt_ );
    tbox_1 = zone_1.BBox( txtbuf );
    zone_1.SetAlignment( FTGL::ALIGN_JUSTIFY );
*/
    glMatrixMode( GL_MODELVIEW );
    glPushMatrix();

//    printf( " d_tx1_ = %f, d_ty1_ = %f, d_tw1_ = %f, d_th1_ = %f\n", d_tx1_, d_ty1_, d_tw1_, d_th1_ );
/*
    tm1x = d_tsc_*tbox_1.Lower().Xf() + d_tx1_; tm1y = d_tsc_*tbox_1.Lower().Yf() + d_ty1_ - d_tla_;
    tm2x = d_tsc_*tbox_1.Upper().Xf() + d_tx1_; tm2y = d_tsc_*tbox_1.Upper().Yf() + d_ty1_ - d_tla_;

    glColor3f( 0.1, 0.1, 0.4 );
    glBegin( GL_LINE_LOOP );
        glVertex3f( tm1x, tm1y, 0.6 );
        glVertex3f( tm2x, tm1y, 0.6 );
        glVertex3f( tm2x, tm2y, 0.6 );
        glVertex3f( tm1x, tm2y, 0.6 );
    glEnd();


    printf( " size of t box X1,Y1 = %f x %f à X2,Y2 %f x %f\n", tm1x, tm1y, tm2x, tm2y );
    printf( " size of Txt1 : d_tx1_ = %f, d_ty1_ = %f, d_tw1_ = %f, d_th1_ = %f\n",
            d_tx1_, d_ty1_, d_tw1_, d_th1_ );

/*
    glBegin( GL_LINE_LOOP );
        glVertex3f( 10.0, 10.0, 0.5 );
        glVertex3f(-10.0, 10.0, 0.5 );
        glVertex3f(-10.0,-10.0, 0.5 );
        glVertex3f( 10.0,-10.0, 0.5 );
    glEnd();
        glVertex3f(-10.0,-10.0, 0.5 );
        glVertex3f( 10.0,-10.0, 0.5 );
    glEnd();


*/
    glTranslatef( d_tx1_, d_ty1_ - d_tla_, 0.1 );
    glScalef( d_tsc_, d_tsc_, 1.0 );



    glColor3f( 0.1, 0.1, 0.9 );

    Layout.Render( i_text_ );

    glPopMatrix();

} // void View_GL::Write_Text().





void Prt_Mat( GLdouble * mat, const char * T )
{
    int i, j, k;

    printf( "\n Matrix %s\n", T );
    k = 0;
    for(i = 0; i< 4; i++) {
        printf( "   " );
        for(j = 0; j<4; j++) {
            printf( " %f", mat[k++] );
        }
        printf( "\n" );
    }
}



void View_GL::draw()
// called to re-draw the GL window.
//
{
    const double dlt = 0.001;
    GLint    whwid,      whhei, // Half display size in image pixels.
             i_mapx,    i_mapy, // PixelMap origine on Display.
             cur_x1,    cur_x2, // Current limits of the displayed image.
             cur_y1,    cur_y2;

    GLenum              maptyp;
    GLdouble rx_scale, ry_scale;

#ifdef  GL_DEBUG
    static GLdouble ProMat[16], ModMat[16];
    static GLdouble xxo, yyo, zzo, xlm, ylm, zlm;
    static GLint ierr, VWP[4];
#endif

//  printf( " Draw Begin!!\n" ); fflush( stdout );
//if (Win_Init) Viewer_Init();

#ifndef MESA
    glDrawBuffer(GL_FRONT_AND_BACK);
#endif // !MESA

    if (!valid()) {             // GL Init is done.
        // Set the display pixel origine on the GL Window center in a 3D space.
        // glOrtho( (GLfloat)-w_ghw_, (GLfloat)w_ghw_, (GLfloat)-ghh_, (GLfloat)w_ghh_, -20.0, 20.0 );
        valid( 1 );
    }

    Set_Format();                    // Outline computing of GL space.
    Set_Geometry();                  // Adapt the Text and imlage disposition.

    glClearColor( 1.0, 1.0, 1.0, 0.0 );  // The color of an empty GL space must be white.
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    // Define the showed image space.
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity();
    glViewport( w_glx_,w_gly_,w_glw_,w_glh_ );  // Set the GL Viewport.
    glOrtho( -d_hwi_-dlt, +d_hwi_+dlt, -d_hhe_-dlt, +d_hhe_+dlt, -1.0, 1.0 );

    glEnable( GL_LINE_SMOOTH );
    glEnable( GL_POLYGON_SMOOTH );

#ifdef GL_DEBUG
    printf( "\n glViewport %d, %d, %d, %d\n", w_glx_, w_gly_, w_glw_, w_glh_ );
    printf( " glOrtho  %f,  %f,  %f,  %f\n\n", -d_hwi_-dlt, +d_hwi_+dlt, -d_hhe_-dlt, +d_hhe_+dlt );

#endif

    glEnable( GL_DEPTH_TEST );
    glMatrixMode( GL_MODELVIEW );
    glLoadIdentity();



    if (1) {
        // Display the document surface limits (the frame).
        glLineWidth( 5.0 );
        glColor3f( 0.4, 0.4, 0.4 );
        glBegin( GL_LINE_LOOP );
            glVertex3f( -d_hwi_, -d_hhe_,  0.9 );
            glVertex3f( +d_hwi_, -d_hhe_,  0.9 );
            glVertex3f( +d_hwi_, +d_hhe_,  0.9 );
            glVertex3f( -d_hwi_, +d_hhe_,  0.9 );
        glEnd();
    }

//printf( "Clear: map %d, mouseSt %d\n", imgmap_!=0, MouseStat ); fflush( stdout );
    if (i_map_) {

#ifdef GL_DEBUG
        if (!MouseStat) {
            printf( "\n ****\n DR Window sizes (screen pixels) : %d, %d\n", w_glw_, w_glh_ );
            printf( " DR Half image sizes  (image  pixels) : %d, %d\n", i_hwi_, i_hhe_ );
            printf( " DR Image Limits ; x [%d..%d], y [%d..%d]\n", i_mx1_, i_mx2_, i_my1_, i_my2_ );
            fflush( stdout );
        }
#endif

        if (ZoomStatus) {
            // Executed after a success full Zoom mouse command.

#ifdef GL_DEBUG
            printf( " DR Zoom Half showed sizes (image  pixels) : %d, %d\n", i_zsw_, i_zsh_ );
            printf( " DR Zoom Center of showed zoom (image  pixels) : %d, %d\n", i_zcx_, i_zcy_ );
            fflush( stdout );
#endif

//          FirstView = 0;
            i_cex_ += i_zcx_; i_cey_ += i_zcy_;
            i_shw_ = i_zsw_; i_shh_ = i_zsh_;
            ZoomStatus = 0;
        }

        if (X_shift) { i_cex_ += X_shift; X_shift = 0; }
        if (Y_shift) { i_cey_ += Y_shift; Y_shift = 0; }

#ifdef GL_DEBUG
        if (!MouseStat) {
            printf( " DR Half showed sizes (image  pixels) : %d, %d\n", i_shw_, i_shh_ );
            fflush( stdout );
        }
#endif

        // Compute the Image scale (in screen_Pixels/Image_Pixels).
        if (!SetScaVal) {
            rx_scale = (GLdouble)di_hwi_/i_shw_; ry_scale = (GLdouble)di_hhe_/i_shh_; /// Modif for Geo d_ -> di_
            r_scale_ = (rx_scale < ry_scale)? rx_scale : ry_scale;
            if (/* FirstView&& */ (r_scale_ > 1.0)) r_scale_ = 1.0;
        } else SetScaVal = 0;


//      W_zoom->value( r_scale_ );

        // Get the window half sizes in image pixels.
        whwid = di_hwi_/r_scale_; whhei = di_hhe_/r_scale_;

#ifdef GL_DEBUG
        if (!MouseStat) {
            printf( " DR scale = %10.6f\n", r_scale_ );
            printf( " DR Half Display sizes (Image pixels) : %d, %d\n", whwid, whhei );
            printf( " DR Part of image (%d, %d), Half display Size (%d, %d)\n", i_cex_, i_cey_, whwid, whhei );
            fflush( stdout );
        }
#endif

        // Get the future visible image part rectangle limits (in image pixels space).
        cur_x1 = i_cex_ -  whwid; cur_y1 = i_cey_ -  whhei;
        cur_x2 = i_cex_ +  whwid; cur_y2 = i_cey_ +  whhei;

        // Compute the image pixel and image row to skip when required.
        if (i_mx1_ < cur_x1) { i_spix = cur_x1 - i_mx1_; i_mapx = cur_x1; }
                        else { i_spix = 0; i_mapx = i_mx1_;}
        i_npix = ((i_mx2_ > cur_x2)? cur_x2 : i_mx2_) - i_mapx;

        if (i_my1_ < cur_y1) { i_srow = cur_y1 - i_my1_; i_mapy = cur_y1; }
                        else { i_srow = 0; i_mapy = i_my1_; }
        i_nrow = ((i_my2_ > cur_y2)? cur_y2 : i_my2_) - i_mapy;

        W_xscroll->value( i_spix, i_npix, 0, i_wid_ );
        W_yscroll->value( i_hei_ -i_srow - i_nrow, i_nrow, 0, i_hei_ );

        // Here i_spix is the number of image pixels to skip in a row, ...
        // ...  i_srow is the number of image rows to skip in the full image, ...
        // ...  i_npix is the number of row pixels to display, ...
        // ...  i_nrow is the number of rows to display.
        // curr_x1.. curr_x2 is the range

//      di_mpx_ = i_mapx*r_scale_ - di_cex_;
//      di_mpy_ = i_mapy*r_scale_ - di_cey_;
//      di_mpx_ = (i_mapx - i_cex_)*r_scale_ + di_cex_;
//      di_mpy_ = (i_mapy - i_cey_)*r_scale_ + di_cey_;
//      di_mpx_ = i_mapx*r_scale_; di_mpy_ = i_mapy*r_scale_;

//      di_mpx_ = di_cex_ - di_hwi_; di_mpy_ = di_cey_ - di_hhe_;

//  printf( "Draw: Image origine GL coord  : %f, %f\n", di_mpx_, di_mpy_ );

#ifdef GL_DEBUG
        if (!MouseStat) {
            printf( " DR *** GL ViewPort : (%d, %d)\n", w_glw_, w_glh_ );
            printf( " DR *** Image space Rectangle : %d, %d, %d, %d\n", cur_x1, cur_y1, cur_x2, cur_y2 );
            printf( " DR *** Scale = %8.5f, MapSizes = (%d, %d)\n", r_scale_, i_npix, i_nrow );
            printf( " DR *** Skip Pixels,Rows (%d, %d), MapOrg. = (%d, %d)\n", i_spix, i_srow, di_mpx_, di_mpy_ );
            fflush( stdout );
        }
#endif


#ifdef GL_DEBUG
        glGetDoublev( GL_PROJECTION_MATRIX, ProMat ); Prt_Mat( ProMat, "Matrice de Projection" );
        glGetDoublev( GL_MODELVIEW_MATRIX, ModMat ); Prt_Mat( ModMat, "Matrice de Modelisation" );
        glGetIntegerv( GL_VIEWPORT, VWP );
        ierr = gluProject( (GLdouble)i_mx1_, (GLdouble)i_my1_, 0.0, ModMat, ProMat, VWP,  &xxo, &yyo, &zzo );
        printf( " DRC err = %d, Win Coord img l,d in window: %f, %f\n", ierr, xxo, yyo );
        ierr = gluProject( (GLdouble)i_mx2_, (GLdouble)i_my2_, 0.0, ModMat, ProMat, VWP, &xlm, &ylm, &zlm );
        printf( " DRC err = %d, Win Coord img r,u in window: %f, %f\n", ierr, xlm, ylm );
#endif


#ifdef GL_DEBUG
        printf( " DR draw() U %d\n", i_pcs_ ); fflush( stdout );
#endif

        switch (i_pcs_) {
            case 1: maptyp = GL_LUMINANCE; break;
            case 3: maptyp =       GL_RGB; break;
            case 4: maptyp =      GL_RGBA; break;

        default:
            fl_alert( "View_GL: Bad type of imag: %d byte/pixel.", i_pcs_ );
            maptyp = -1;
            printf( " Bad type of image\n" ); fflush( stdout );
//          exit( 2 );
        }

#ifdef GL_DEBUG
        printf( " draw() V %d\n", maptyp ); fflush( stdout );
#endif

        if (maptyp>0) {

#ifdef GL_DEBUG
            printf( "-----> i_widt=%d, i_spix=%d, i_srow=%d, at (%d,%d)\n", i_wid_,i_spix,i_srow, di_mpx_, di_mpy_ );
//          fflush( stdout );
#endif

            glPixelStorei( GL_UNPACK_ROW_LENGTH,    i_wid_ );
            glPixelStorei( GL_UNPACK_SKIP_PIXELS,   i_spix );
            glPixelStorei( GL_UNPACK_SKIP_ROWS,     i_srow );
            glPixelStorei( GL_UNPACK_ALIGNMENT,          1 );
            glRasterPos3f( di_mpx_+0.0001, di_mpy_+0.0001, 0.0 );
//          GLboolean valid; glGetBooleanv( GL_CURRENT_RASTER_POSITION_VALID, &valid );
//          printf( " PIxmap Position valid = %d\n", valid );
            glPixelZoom( r_scale_, r_scale_ );
            glDrawPixels(  i_npix, i_nrow, maptyp, GL_UNSIGNED_BYTE, (GLvoid*)i_map_ );

/*
            glColor3f( 0.8, 0.2, 0.2 ); glLineWidth( 2.0 );
            glBegin( GL_LINE_LOOP );
                glVertex3f(        d_tx1_,        d_ty1_,  0.8 );
                glVertex3f(        d_tx1_, d_ty1_-d_th1_,  0.8 );
                glVertex3f( d_tx1_+d_tw1_, d_ty1_-d_th1_,  0.8 );
                glVertex3f( d_tx1_+d_tw1_,        d_ty1_,  0.8 );
            glEnd();

            if (d_tw2_) {
                glColor3f( 0.2, 0.2, 0.8 ); glLineWidth( 2.0 );
                glBegin( GL_LINE_LOOP );
                    glVertex3f(        d_tx2_,        d_ty2_,  0.8 );
                    glVertex3f(        d_tx2_, d_ty2_-d_th2_,  0.8 );
                    glVertex3f( d_tx2_+d_tw2_, d_ty2_-d_th2_,  0.8 );
                    glVertex3f( d_tx2_+d_tw2_,        d_ty2_,  0.8 );
                glEnd();
            }
/*
*/

            Write_Text();

/*
#ifdef GL_DEBUG
        glLineWidth( 5.0 );
        glColor3f( 0.2, 0.3, 0.8 );
        glBegin( GL_LINE_LOOP );
            glVertex3f( -10.0, -10.0,  0.8 );
            glVertex3f( +10.0, -10.0,  0.8 );
            glVertex3f( +10.0, +10.0,  0.8 );
            glVertex3f( -10.0, +10.0,  0.8 );
        glEnd();
        glColor3f( 0.8, 0.3, 0.2 );
        glBegin( GL_LINE_LOOP );
            glVertex3f( di_mpx_-10.0, di_mpy_-10.0,  0.8 );
            glVertex3f( di_mpx_+10.0, di_mpy_-10.0,  0.8 );
            glVertex3f( di_mpx_+10.0, di_mpy_+10.0,  0.8 );
            glVertex3f( di_mpx_-10.0, di_mpy_+10.0,  0.8 );
        glEnd();
#endif
*/
        }

#ifdef GL_DEBUG
        printf( " DR draw() W\n" ); fflush( stdout );
#endif
        if (MouseStat) {
            glLineWidth( 2.0 );
            glColor3f( 0.7, 0.7, 0.7 );
            glBegin( GL_LINE_LOOP );
            glVertex3f( w_zx1_, w_zy1_,  0.5 );
            glVertex3f( w_zx2_, w_zy1_,  0.5 );
            glVertex3f( w_zx2_, w_zy2_,  0.5 );
            glVertex3f( w_zx1_, w_zy2_,  0.5 );
            glEnd();
        }
    }

#ifdef GL_DEBUG
    printf( " DR draw() X\n" ); fflush( stdout );
#endif

#ifndef MESA
    glDrawBuffer(GL_BACK);
#endif // !MESA

    if (Fl::visible_focus())take_focus();

//printf( " Draw Process End\n" ); fflush( stdout );
} // void View_GL::draw().


//
// end of <PW-Id>.
//




