//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module to manage the Image file for OpenGL.
//

#include <stdio.h>
#include <GL/gl.h>

#include <FL/fl_utf8.h>

#include "ImageMan_MAP.h"

#include "ImageMan_JPG.h"
#include "ImageMan_PNG.h"
#include "ImageMan_BMP.h"



Image_MAP::Image_MAP()
// Empty image map creator.
//
{
    w_ = h_ = p_ = e_ = 0;
    ide_ = 0; all_ = 0;
    map_ = NULL;
}



GLubyte * Image_MAP::New_Map( GLint nsz )
// Allocate an image map when :
//   - The current map_ does not exist or,
//   - the current map_ is too small or,
//   - the current map_ is too large of more than 20%.
//
{
    if (nsz>all_||(nsz<all_&&((all_-nsz)>all_/20))) {
        if (map_) delete[] map_;
        if  (nsz>0) map_ = new GLubyte[nsz];
        all_ = nsz;
        e_ = 0;
    }
    return map_;
}



int Image_MAP::Read( const char * filename )
// Function Load or Reload an image in the image cache,
// from the specified file.
// On succes 1 is returned and 0 otherwise.
//
{
    FILE          * fp;
    uchar   header[64];
    int        ie,  ii;

    ie = 0;     // Assume that no image is loaded.
    for(ii=0;ii<16;ii++) header[ii] = 0;
    if (fp = fl_fopen( filename, "rb" )) {
        if (fread( header, 1, sizeof( header ), fp)) { /* Ignore Read error */ }
        fclose( fp ); // Ignore read error.
    } else return 0;
    ie = 1; // Assume success.
    if (memcmp( header, "BM", 2 ) == 0)  // BMP file.
        this->Get_BMP( filename );
    else
    if (memcmp( header, "\377\330\377", 3 ) == 0 )  // JPEG file.
        this->Get_JPG( filename );
    else
    if (memcmp( header, "\211PNG", 4 ) == 0)
        this->Get_PNG( filename );
    else {
        E( ERR_FORMAT );
        ie = 0;  // Unsuported image format ! 
    }
    if (E()<0) ie = 0;
    return ie;
}



int Image_MAP::Load( const uchar * data )
// Function Load or Reload an image in the image cache,
// from the specified memory data block.
// On succes 1 is returned and 0 otherwise.
// Warning : The BMP format is not supported.
//
{
  uchar   header[64];
  int        ie,  ii;
  static union {
    uchar cval[sizeof( int )];
    int  size;
  } equ;

  // Get the size of image data block.
  for(ii=0;ii<sizeof( int );ii++) equ.cval[ii] = *(data++);

  // Get the image header (to find the appropriate format).
  for(ii=0;ii<16;ii++) header[ii] = data[ii];
  ie = 1; // Assume success.
  if (memcmp( header, "\377\330\377", 3 ) == 0 )  // JPEG file.
    this->Get_JPG_MEM( data );
  else
    if (memcmp( header, "\211PNG", 4 ) == 0)
      this->Load_PNG_( NULL, data, equ.size );
    else {
      E( ERR_FORMAT );
      ie = 0;  // Unsuported image format ! 
    }
  if (E()<0) ie = 0;
  return ie;  
}


//
// end of <PW-Id>.
//


