//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module to manage the Image file/Pixel Map for OpenGL.
//


#include <stdlib.h>
#include <string.h>

#include "ImageMan_REF.h"


//
// Define  the flag of the bit#15 in the 16 bits flag word ...
// ... as the flag of memory image.
//
#define MEMIMG_FLG 0x8000



// Definition of four possible partitions of the image cache.
// The Fixed limit to 4 is probably definitive for this caching system
// using integer (32 or 64 bits) to identify each image.
// with this limit that must a power

static GLint        CacheLngt   =    0, // The Allocated size of the image cache.
                    CacheSize   =    0; // The current size of image cache and ...
static GLuint       CacheMask   =    0, // ... its related mask.
                    CachePart   =    0; // Number of possible view(s).

static char*        CurrSDir    = NULL; // Pointer to the current directory to use if defined.

static Image_MAP  * CurrEnt     = NULL, // Pointer to the selected cache entry.
                  * ImgCache    = NULL, // Pointer to the whole Cache.
                 ** CacheBtb    = NULL, // Pointer to the allocated cache image by partition #.
                 ** CacheLnk    = NULL; // Pointer to the allocated cache image by logical #.



// We define the image cache to keep the last acceded Pixel MAP.
// This Cache allows to avoid doing to reload already loaded
// images without to use an exessive amount of memory.
// For RGB photography 128 Slides of 12Mpixels use ~1.43Gb.
//
// This cache can be shared between several (maximum 4) partition
// by the CacheSplit() function, and re-merged by the CacheMerge() function.
// The first call of CacheMerge split the current cache in two sub-cache.
//




void Image_REF::Image_Init( GLint siz )
//
// This Routine must be call before any others operations ...
// ... to create the image memory cache.
//
{
    GLint esz = 2;

    if (CacheSize)      return; // Nothing to do if the cache is OK.

    // Build the cache for one partition.
    while (esz<siz) esz  *=  2; // Find the minimal power of 2 >= siz.
    CacheLngt =            esz; // Set the cache allocated size and init ...
    CacheSize =      CacheLngt; // ... the dynamic cache size.
    CacheMask =        esz - 1; // ... related mask.
    CachePart =              1; // Only one cache partition, we ca have only one GL view.

    ImgCache  = new Image_MAP[esz];     // Create the Image cache (initialized by Image_REF).
 
    CurrSDir  =           NULL; // By default, the image filename are full paths.
    CurrEnt   =       ImgCache; // Current pointer is set to never be NULL.

    CacheBtb  = new Image_MAP*[MaxCachePart];
    CacheLnk  = new Image_MAP*[MaxCachePart];
    CacheLnk[0] = CacheBtb[0] = ImgCache;

    for(int i = 1; i < MaxCachePart; i++)
        CacheBtb[i] = CacheLnk[i] = NULL;
} // void Image_REF::Image_Init( GLint siz ).



void Image_REF::ClearCache()
{
    // Erase all caches Image_MAP identiers to force the ...
    
    for(int id=0; id<CacheLngt; id++) {
        ImgCache[id].Ide( 0 );
    }
    
} // void Image_REF::ClearCache().



void Image_REF::CacheSplit( int uid )
// uid must be the smallest image ref. user.
{
    int i, np;

//printf( " Split A %d\n", uid ); fflush( stdout );
    np = 0;                                     // Evaluate the number of used table entry.
    for( i = 0; i < MaxCachePart; i++) if (CacheLnk[i]) np++;

    if (np >= CachePart) {                      // When the number of cache partition is too low ...
        CachePart *= 2;                         // ... we increase it ( *2 ), ...
        CacheSize  = CacheLngt/CachePart;       // ... set the new possible number of view.
        CacheMask  = CacheSize - 1;             // Set the new cache identifier mask.

        for(i = 1; i < CachePart; i++)          // Build the new partition bases table.
            CacheBtb[i] = ImgCache+i*CacheSize; // The first entry does not change.
    }
    // Reassign the partition for the previously existing.
    np = 0;
    for(i = 0; i < MaxCachePart; i++)
        if (CacheLnk[i]) CacheLnk[i] = CacheBtb[np++];

    // Assign a partition for the new partition request.
    CacheLnk[uid] = CacheBtb[np];               // Assign the first free partition.
    
    ClearCache();                               // ... and clear all cache to force new reload.
} // void Image_REF::CacheSplit().



void Image_REF::CacheMerge( int uid )
{
    int i, np;

    CacheLnk[uid] = NULL;               // Free the link of the user image access.

    np = 0;
    // We examine the base cache table to determine the number of unused partitions.
    for(int i=0;i<MaxCachePart;i++) if (CacheLnk[i]) np++;

    if (np <= CachePart/2) {            // We can decrease the nember of partition.
        CachePart /= 2;                 // We must change the Display numbers.
        CacheSize = CacheLngt/CachePart;// Set the new possible number of view.
        CacheMask  = CacheSize - 1;     // Set the new cache identifier mask.

        // Build the new partition table.
        for(i = 0; i < CachePart; i++)
            CacheBtb[i] = ImgCache + i*CacheSize;
    }
    // Re-assign a partition to each image user.
    np = 0;
    for(int i = 0; i < MaxCachePart; i++)
        if (CacheLnk[i]) CacheLnk[i] = CacheBtb[np++];


    ClearCache();
} // void Image_REF::CacheMerge().



void Image_REF::MapCp2Oth( UInt32 sz )
// Copy the Current Slide Cache Entry to Same image Entries in other cache partitions ...
// ... if these entries exist. Used to update the same image view on several View_UI.
{
    int id  = ident_&CacheMask; // Get the current cache index (inside a cache partition).
    Image_MAP            * img;

    // Scan all Cache partition to clear ident of image copies except for the current ...
    // ... partition to force a new reload and apply the present transformation.
    while (id<CacheLngt) {
        img = ImgCache+id;      // Get the address of the slide cache entry.
        if (ident_ == img->Ide())       // Select it only for the same image ...
            if (img != CurrEnt) {       // ... and not in the current partition.
                img->W( CurrEnt->W() ); img->H( CurrEnt->H() ); // Copy the slide width and high, ...
                img->P( CurrEnt->P() ); img->E( CurrEnt->E() ); // ... the size of pixel and extra/error code, ...
                if (img->Map())
                    memcpy( img->Map(), CurrEnt->Map(), sz );   // ... and the image pixelmap (if already allocated).
                img->E( 1 );
            }
        id += CacheSize;        // Continue to next cache partition.
    }   
} // void Image_REF::MapCp2Oth( IIde_t sz ).



GLubyte * Image_REF::Check( int id )
{
    // id is the window index used to select the related cache part.
    // Get the image cache entry address.
    // If the image is not present in the cache, the image
    // must be loaded or re-loaded.
    CurrEnt = CacheLnk[id] + (ident_&CacheMask);
    if (ident_==CurrEnt->Ide()) return CurrEnt->Map();
                           else return NULL;
} // GLubyte * Image_REF::Check( int id ).



void Image_REF::UnDTrf( Image_Trf trf )
// To transform the Pixel Map when the Width and hight are exchanged.
{
    GLint   ik, x1, y1, r2, sz,
                    ww, hh, pp;
    GLubyte *m1, *m2, *p0, *px,
                          *pxy;
    
    ww = CurrEnt->W();
    hh = CurrEnt->H();
    pp = CurrEnt->P();
    sz = ww*hh*pp;
    m1 = CurrEnt->Map();
    m2 = new GLubyte[ww*hh*pp];

    p0 =    m1; // x1, y1 = 0; p1 -> map1[ 0, 0].
    r2 = hh*pp; // Target row length (hh, ww become the width, high !
    switch (trf) {
        case Op_90: // Rotate from +90° .
            // Rotation Matrix to apply on map[y1,x1]:
            //
            //      ( 0, -1, hh-1 ) (x1)   (hh-1-y1)
            //      ( 1,  0,    0 )*(y1) = (  x1   ) or m2[x1,hh-1-y1] = m1[y1,x1].
            //      ( 0,  0,    1 ) ( 1)   (   1   ) 
            //
            px =   m2 + r2;     // px  -> m2[0,hh] == m1[r2] (column pointer p2).
            for(y1=0;y1<hh;y1++) {
                px -= pp;       // px  -> m2[0,x2] with x2 = hh - 1 - y1.
                pxy = px;       // pxy -> m2[0,x2] == m2[0,hh-1-y1].
                for(x1=0;x1<ww;x1++) {
                    for (ik=0;ik<pp;ik++) pxy[ik] = *(p0++);
                    pxy += r2;  // pxy -> m2[x1,hh-1-y1] == m2[y2,x2].
//                              // m2[y2*r2+x2*pp+k] = *(p1++);
                }
            }
        break;

        case Op270: // Rotate from +90° .
            // Rotation -90° (or +270°) Matrix to apply on map[y1,x1]:
            //
            //      ( 0,  1,    0 ) (x1)   (  y1   )
            //      (-1,  0, ww-1 )*(y1) = (ww-1-x1) or m2[w-1-x1,y1] = m1[y1,x1].
            //      ( 0,  0,    1 ) ( 1)   (   1   ) 
            //
            px =        m2; // px -> m2[0,0].
            for(y1=0;y1<hh;y1++) {
                pxy = px + sz - r2;     // pxy -> m2[ww-1,y1] == m2[y2,x2].
                for(x1=0;x1<ww;x1++) {
                    for (ik=0;ik<pp;ik++) pxy[ik] = *(p0++); // map2[y2*r2+x2*pp+k] = *(p1++);
                    pxy -= r2;
                }
                px += pp;               // x2++ (with pp bytes) px -> m2[0,y1].
            }
        break;

        case OpMiu: // // Mirror v along 1'th diagonal .
            // Transformation Matrix to apply on map[y1,x1]:
            //
            //      ( 0,  1,    0 ) (x1)   (   1   )
            //      ( 1,  0,    0 )*(y1) = (   1   ) or m2[x1,y1] = m1[y1,x1].
            //      ( 0,  0,    1 ) ( 1)   (   1   ) 
            //
            px = m2;
            for(y1=0;y1<hh;y1++) {
                pxy = px;
                for(x1=0;x1<ww;x1++) {
                    for (ik=0;ik<pp;ik++) pxy[ik] = *(p0++);
                    pxy += r2;
                }
                px += pp;
            }                
        break;

        case OpMiv: // Mirror v along 2'th diagonal.
            // Transformation Matrix to apply on map[y1,x1]:
            //
            //      ( 0, -1,    0 ) (x1)   (hh-1-y1)
            //      (-1,  0,    0 )*(y1) = (ww-1-x1) or m2[ww-x1-1,hh-y1-1] = m1[y1,x1].
            //      ( 0,  0,    1 ) ( 1)   (   1   ) 
            //
            px = m2 + sz - pp;          // px-> m2[ww-x1-1,hh-y1-1] with here x1 = y1 = 0. 
            for(y1=0;y1<hh;y1++) {
                pxy = px;               // pxy->m2[ww-x1-1,hh-y1-1] with here x1 = 0.
                for(x1=0;x1<ww;x1++) {
                    for (ik=0;ik<pp;ik++) pxy[ik] = *(p0++); // m2[ww-x1-1,hh-y1-1] = m1[y1,x1].
                    pxy -= r2;
                }
                px -= pp;
            }                
        break;
        
        default: ;
    }
    memcpy( m1, m2, sz );
    delete[] m2;
    CurrEnt->W( hh );
    CurrEnt->H( ww );
    m2 = NULL;
} // Image_Ref::UnDTrf.



Image_REF::Image_REF( const char * filename, IIde_t ident )
// The constructor of Image definition.
{
    int ll;

    if (filename) {
        fname_ =  strdup( filename );   // set the image file path.
        ll = strlen( fname_ );
        if (fname_[ll-1]==DIRSEP) fname_[ll-1] = 0;
    } else fname_ = NULL;
    ident_ =     ident;         // Set the unique integer identifier.
    CuOpe_ = OrOpe_ = Op_No;    // Initialize the orientation fields.
//  uflgs_ = 0;                 // Any user flags for initialization.
} // Image_REF::Image_REF( const char * filename, IIde_t ident ).



Image_REF::Image_REF( const uchar * data, IIde_t ident )
// The constructor of Memory Image definition.
{
    int ll;

    fname_ = (char *) data;     // set the image file path.
    ident_ =     ident;         // Set the unique integer identifier.
    CuOpe_ = OrOpe_ = Op_No;    // Initialize the orientation fields.
//  uflgs_ = MEMIMG_FLG;        // Set the Memory image flag.
} // Image_REF::Image_REF( IIde_t ident, const uchar * data ).



// The destructor.
Image_REF::~Image_REF()
{
    if (fname_) free( fname_ );
    fname_ = NULL; ident_ = 0; CuOpe_ = OrOpe_ = Op_No;
//  uflgs_ = 0;
} // Image_REF::~Image_REF().



void Image_REF::UsedPath( const char * path )
// Note: The routines UsedPath are not reentrant fonctions.
//   Should not be used when parallel tasks use different image directories.
//
// Set or suppress the current Directory to load image files.
// If path has not the final DIRSEP, a DIRSEP is append.
{
    if (path) {
        UInt32 ll = strlen( path );
        if (CurrSDir) {
            if (strlen( CurrSDir ) < ll+2) {
                delete[] CurrSDir;
                CurrSDir = new char[ll+2];
            }
        } else CurrSDir = new char[ll+2];
        strcpy( CurrSDir, path );
        if (CurrSDir[ll-1] != DIRSEP) {
            CurrSDir[ll] = DIRSEP;
            CurrSDir[ll+1] = 0;
        }
    } else
        if (CurrSDir) delete[] CurrSDir;
}



const char* Image_REF::UsedPath() { return CurrSDir; }



GLubyte * Image_REF::Get( const char * fp, int id, int memflg )
{
    int       ie;

    GLubyte * map = this->Check( id );

    if (!map) { // When the image is not present in the cache.

        if (CurrEnt->Ide()) { // An other image was in the cache at this place.
            // We must save the actual Image Context.
            //// We must apply the flagged transformations to the image file.
            
            
        }
        if (memflg) {
            ie = CurrEnt->Load( (uchar*)fp );   // Expend and set the block memory image.
        } else {
            ie = CurrEnt->Read( fp );           // Load the Image in memory.
        }

        if (ie) {
            CurrEnt->Ide( ident_ );             // Set the unique identifier in the cache place.
        }
        if (CurrEnt->E()>=0) {                  // No Loading error.
            if (CuOpe_)
                this->Transform( CuOpe_|OpIni, fp, id );      // Apply registred transformation.
        }
        if (ie) return CurrEnt->Map();
           else return NULL;
    }
    return map;
} // GLubyte * Image_REF::Get( const char * fp, int id, int memflg ) .



GLubyte * Image_REF::Get( int id, int memflg )
{
    char *fpath = NULL;
    int             ie;

    GLubyte * map = this->Check( id );

    if (!map) { // When the image is not present in the cache.
        if (CurrEnt->Ide()) { // An other image was in the cache at this place.
            // We must save the actual Image Context.
            //// We must apply the flagged transformations to the image file.
            
            
        }

        if (memflg)
            ie = CurrEnt->Load( (uchar*)fpath );// Expend and set the block memory image.
        else {
            if (CurrSDir) {                     // if a directory is specified ...
                int l1 = strlen( CurrSDir ),    //   ... create a complet image file ...
                    l2 = l1 + strlen( fname_ ); //   ... path in string fpath.
                fpath = new char[l2+1];
                strcpy( fpath, CurrSDir );
                strcpy( fpath+l1, fname_ );
                ie = CurrEnt->Read( fpath );    //   Get and Load the Image in memory.
                delete[] fpath;                 //   Free the fpath memory.
            } else                              // else (fname_ must be a complet path.
                ie = CurrEnt->Read( fname_ );   //   Load the Image in memory.
        }
        if (ie) {
            CurrEnt->Ide( ident_ );             // Set the unique identifier in the cache place.
        }
        if (CurrEnt->E()>=0) {                  // No Loading error.
            if (CuOpe_)
                this->Transform( CuOpe_|OpIni, fpath );      // Apply registred transformation.
        }
        if (ie) return CurrEnt->Map();
           else return NULL;
    }
    return map;
} // GLubyte * Image_REF::Get( int id, int memflg ).



/// void  Image_REF::Release()
///{
///    GLubyte *map = this->Check();
///    CurrEnt->Ide( 0 );
///}


GLint Image_REF::W() { return CurrEnt?CurrEnt->W():0; }
GLint Image_REF::H() { return CurrEnt?CurrEnt->H():0; }
GLint Image_REF::P() { return CurrEnt?CurrEnt->P():0; }
GLint Image_REF::E() { return CurrEnt?CurrEnt->E():0; }

GLubyte * Image_REF::Map() { return CurrEnt?CurrEnt->Map():NULL; }


int  Image_REF::Transform( Image_Trf trf, const char * fpath, int id )
// Perform the image transformations.
{
    static Image_Trf grprel[8][8] = { // Plane Groupe 422 : Product table. 
        //  e      4      2     4^3     mx     my     mu     mv
        { Op_No, Op_90, Op180, Op270, OpMix, OpMiy, OpMiu, OpMiv }, //  e
        { Op_90, Op180, Op270, Op_No, OpMiv, OpMiu, OpMix, OpMiy }, //  4
        { Op180, Op270, Op_No, Op_90, OpMiy, OpMix, OpMiv, OpMiu }, //  2
        { Op270, Op_No, Op_90, Op180, OpMiu, OpMiv, OpMiy, OpMix }, //  4^3
        { OpMix, OpMiu, OpMiy, OpMiv, Op_No, Op180, Op_90, Op270 }, //  mx
        { OpMiy, OpMiv, OpMix, OpMiu, Op180, Op_No, Op270, Op_90 }, //  my
        { OpMiu, OpMiy, OpMiv, OpMix, Op270, Op_90, Op_No, Op180 }, //  mu
        { OpMiv, OpMix, OpMiu, OpMiy, Op_90, Op270, Op180, Op_No }  //  mv

    };
    int iniflg, upd;

    if (trf&OpIni) { trf &= ~OpIni; iniflg = 1; }
              else iniflg = 0;
    if (trf==Op_No) {
        // Return the image in the original state.
        if (CuOpe_== Op_90) trf = Op270;
        else if (CuOpe_ == Op270) trf = Op_90;
        else trf = CuOpe_;
        if (OrOpe_ != Op_No) trf = grprel[OrOpe_][trf];
    }
    if (trf!=Op_No) {
        int         ww, hh, pp,
                rs, sz, ih, ik;
        GLubyte *map, *p1, *p2;
        GLubyte          tm[4];
        
        map = this->Check( id );

        if (!map) map = this->Get( fpath, id );

        if (map) {
            ww = CurrEnt->W();
            hh = CurrEnt->H();
            pp = CurrEnt->P();
            rs = ww*pp;
            sz = hh*rs;
            p1 = map;
            switch (trf) {
                case Op180: // 180° rotation.
                    p2 = map + sz - pp;
                    while (p2>p1) {
                        for(ik=0;ik<pp;ik++) tm[ik] = p1[ik];
                        for(ik=0;ik<pp;ik++) p1[ik] = p2[ik];
                        for(ik=0;ik<pp;ik++) p2[ik] = tm[ik];
                        p1 += pp;
                        p2 -= pp;
                    }                
                break;

                case OpMix: // Mirror along Y.
                    for(ih=0;ih<hh;ih++) {
                        GLubyte *pp1 = p1, *pp2 = p1 + rs - pp;
                        while (pp2>pp1) {
                            for(ik=0;ik<pp;ik++) tm[ik]  = pp1[ik]; 
                            for(ik=0;ik<pp;ik++) pp1[ik] = pp2[ik];
                            for(ik=0;ik<pp;ik++) pp2[ik] =  tm[ik];
                            pp1 += pp;
                            pp2 -= pp;
                        }
                        p1 += rs;
                    }
                break;

                case OpMiy: {
                    // Mirror along X.
                    GLubyte * rw = new GLubyte[rs];
                    p1 = map; p2 = map + sz;
                    while (p1<p2) {
                        p2 -= rs;
                        memcpy( rw, p1, rs );
                        memcpy( p1, p2, rs );
                        memcpy( p2, rw, rs );
                        p1 += rs;
                    }
                    delete[] rw;
                }
                break;
                
                default:
                    UnDTrf( trf );
            }
            if (!iniflg) CuOpe_ = grprel[trf][CuOpe_];
            if (CachePart > 1) MapCp2Oth( sz );
        }
    }
    // Set return value to signal when a transformation must be applied when image is loaded :
    //   0  No change and no transformation to apply (No context allowed).
    //   1  A transformation to apply, but without change for context.
    //   2  No transformation to apply, but it is change from previous context.
    //   3  A transformation to apply and must be set in the context.
    //
    upd = (CuOpe_)? 1 : 0;
    if (CuOpe_ != OrOpe_) upd += 2;    // Signal a transf. is set but not changed. 
    return upd;
} // int  Image_REF::Transform( Image_Trf trf, const char * fpath, int id ).


//
// end of <PW-Id>.
//
