//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers DiaViewer Software
//

//
// Module  to analyse a directory tree.
//
//


#include <sys/stat.h>
#include <stdio.h>
#include <dirent.h>
#include <errno.h>
#include <string.h>
#include <stdarg.h>


//#define  SERVICE_DIR_M

#include "Service_DIR.h"

#include "Service_ENV.h"
#include "Service_RE8STK.h"
#include "Service_STACK.h"




#define MAX_ERROR   16
#define QSORT_LIMIT 16

static const int MaMi = 'a' - 'A';
static int               gbl_ierr;



SrvDir::SrvDir() {
    typls_ = 0; fkey_ = 0; nerr_ = flgs_ = 0; nlvl_ = -1;
    // We use the less argument stack constructor without argument then ...
    // we must complete it here because we cannot call sizeof() ...
    // ...  in a declarative sequence.

    filtab_.Setup( 512, &fitb_, &fisp_ );
    dirtab_.Setup( 128, &ditb_, &disp_ );
    dintab_.Setup( 128, &dntb_, &dnsp_ );

    ndstk_.Fix_Init( sizeof( DirNode), sizeof( ndtab_ ),
                     (UInt_8*)&ndtab_, (UInt_8**)(&ndsp_) );


} // SrvDir::SrvDir().



SrvDir::~SrvDir() {
    typls_ = 0; fkey_ = 0; path_.Clear();
//  dstk_.Clean(); fstk_.Clean();
} // SrvDir::SrvDir::~SrvDir().



int SrvDir::ErrMan( int ie, const char * msg )
// Default virtual routine to use without child class.
{
    if (ehdl_) ie = ehdl_( ie, msg ); else  fprintf( stderr, "%s\n", msg );
    return ie;
} // int SrvDir::ErrMan(int nk, const char * msg ).


// Scan Error Manager.
// Value of error number :
//     0 : No error.
//   > 0 : Error but we continue the scan.
//  = -1 : No Error but stop the scan.
//  < -1 : Fatal error => The scan is stopped.
//
int  SrvDir::Error_Manager( int ie, const char * frm, ... ) {
    va_list                 ap;
    int                      n;
    char   buf[512],  fmt[128];
    const char * mne = "SrvDir : To Many (%d) Error => Stop scan on Fatal errors.";

    snprintf( fmt, 127, " *** SrvDir%sError %3d : %s", (ie < 0)? " Fatal ": " ", ie, frm );

    va_start( ap, frm );
    ::vsnprintf( buf, 511, fmt, ap );
    va_end( ap );

    ie = ErrMan( ie, buf );

    if (ie > 0 && ++nerr_ > MAX_ERROR) {
        snprintf( buf, 511, mne, MAX_ERROR );
        ie = ErrMan( -2, buf );
    }
    return ie;
} // int  SrvDir::Error_Manager( int ie, const char * frm, ... ).





int SrvDir::StartDir( const char * nam )
// Default virtual routine to use without child class.
//
// The returned value must be :
//   >  0 : Scan the directory.
//   =  0 : Skip the director.
//   <  0 : Stop the tree scan.
//
{
    if (dibg_) return dibg_( nam ); else return 0;
} // int SrvDir::StartDir( const char * msg ).


int SrvDir::EndDir( int nd, int ns )
// Default virtual routine to use without child class.
//
// The returned value must be :
//   >= 0 : Continue the tree scan.
//   <  0 : Stop the tree scan.
//
{
    if (dien_) return dien_( nd, ns ); else return 0;
} // int EndDir::ErrMan(int nk, const char * msg ).




int SrvDir::SelFile( int nk, const char * name )
// Default virtual routine to use without child class.
//
// The returned value must be :
//   >= 0 : Continue the tree scan.
//   <  0 : Stop the tree scan.
//
{
    if (rfil_) return rfil_( nk, name ); else return 0;
} // int SrvDir::SelFile(int nk, const char * msg ).




void SrvDir::SetFltExt( const FTypLst extb ) {

    char         * str;
    char            ch;
    int   ii,  jj,  ln;

    ii = ln = 0;
    while ((extb[ln])) ln++;  // Evaluate the size of extensions size.
    if (typls_) {
        while ((str = typls_[ii++])) { if (str) delete[] str; }
        delete[] typls_;
    }
    typls_ = new char *[ln+1];
    for(ii = 0; ii < ln; ii++)
        typls_[ii] = strdup( extb[ii]);
    typls_[ln] = 0;
} // void SrvDir::SetFltExt( DirFiltre_t flt ).



// Small routine to reject any directory entries as "." or "..".
int  SrvDir::Check_FName( const char * nm ) {
    int ok = 1;

    if (nm && nm[0]) {
        if (nm[0] == '.') {
            if (!nm[1] || (nm[1] == '.' && ! nm[2])) ok = 0;
        }
    } else ok = 0;
    return ok;
} // int  SrvDir::Check_FName( const char * nm ).



// Small Filter member to check matching of filename with accepted filetypes.
// A filter function must be return a value >= 0 to select a file, ...
// ... A negative to unselect it. But a positive value will be transmitted to ...
// ... to the SelFile virtual function in its first argument.
// The first kn argument should be 0 for any file and 1 directory.
// In the case of directory, the return value should be in the range 1..127 for file ...
// ... and 0..127 or 128..255 for directory. In the first case, the directory ...
// ... is put in the directory to scan and in the second one, it is put with file ..
// ... to be keep as the target of search.
//
int  SrvDir::Filter( int kn, const char * nm ) {
    const char * p = nm + strlen( nm );
    char  ch, buf[16];
    int ii = 0;

    if (!kn) return 0;                  // No filter for directory.

    while (p != nm && (*p) != '.') p--; // Loop to find the last period of nm.
    if (p == nm && p[0]) p = "?";       // '?' is used to design a null extension.
    if (p[1]) p++;                      // for filename end by '. it is the extension.
    ii = 0;
    while ((ch = *(p++)) && ii < 15) {
        if (TstFlg( Dir_NCase )) { if ((ch >= 'A') && (ch <= 'Z')) ch += MaMi; }
        buf[ii++] = ch;
    }
    buf[ii] = 0;
    ii = 0;
    // Loop to check for each file extension.
    while (typls_[ii]) {
        if (!strcmp( typls_[ii], buf )) break;
        ii++;
    }
    return (typls_[ii]) ? ii : -1;
} // int SrvDir::Filter( const char * nm ).



//
// Sorting private members function.
//

//
// With Pascal, the FDRef/DiRef the record variant should be used.
//
SrvDir::FDAcc SrvDir::CreateRef( int fd, int kn, const char * nm )
{
    FDRef      rec;
    int    ii, len;

    rec.knd_  = kn;                     // Load the object code (dir of file type).
    if (fd) rec.knd_ |= 0x8000;         // Add the directory flag when required.
    if (kn && nm &&nm[0]) {
        len = strlen( nm ) + 1;         // Do not forget the null terminator of name !
        for(ii = 0; ii < len; ii++) rec.nam_[ii] = nm[ii];
    } else len = 0;
    len += sizeof( SInt16 );            // Add to length the kind field size, ...
    return fddref_.Push( &rec, len );   // ... push the FDRef in the FDref stack and ...
} // SrvDir::FDAcc SrvDir::CreateRef( int fd, int kn, const char * nam ).



SrvDir::DNAcc SrvDir::CreateDNode( FDAcc ref, int nd, int nf, int cnd, int cnf,
                                   FDAcc * fil, DNAcc * dil )
{
    DirNode        rec;
    int     ii, l1, l2;

    rec.ref_    =          ref; // The object is always a directory.
    rec.ditb_   =          dil;
    rec.fitb_   =          fil;
    rec.ndir_   =           nd; // Set the number of directories and ...
    rec.nfil_   =           nf; // ... files entries.
    rec.cndir_  =     cnd + nd;
    rec.cnfil_  =     cnf + nf;
    return dnodem_.Push( rec );
} // SrvDir::FDRef * SrvDir::CreateDNode( int kn, int nd, int nf, const char * nm ).



void SrvDir::FDRef_SimplSort( FDRef ** tb, int min, int max )
// Warning : We sort in decreasing order to get increasing with stack Pop function.
{
    int       jj;
    FDRef   * id;

    for(int ii=min+1; ii<=max; ii++) {
        id = tb[ii];
        jj = ii - 1;
        while ((jj >= min) && (DFREF_NAME_CMP( DFRefName( (tb[jj]) ), DFRefName( id ) ) < 0)) {
            tb[jj+1] = tb[jj];
            jj--;
        }
        tb[jj+1] = id;
    }
} // void SrvDir::FDRef_SimplSort( FDRef *tb, int min, int max ).



void SrvDir::FDRef_QuickSort( FDRef ** tb, int min, int max )
// Perform a Quick sort.
// Warning : We sort in decreasing order to get increasing with stack Pop function.
{
  int        ic,    ip;
  FDRef    * pi,  * tm;

  if (min>=max) return;

  ip = (min + max)/2;   // Get the middle Sort table index ...
  pi =        tb[ip];   // ... as the pivot.

  tb[ip] = tb[min];     // Put the pivot item in first position.
  tb[min] = pi;         //

  ic = min;             //
  for(int ii = min; ii <= max; ii++) {
    // Loop on all entries.
    if (DFREF_NAME_CMP( DFRefName( tb[ii] ), DFRefName( pi ) ) > 0) {
      if (ii != ++ic) {
        // When the entry are differents, we exchange them.
        tm = tb[ii]; tb[ii] = tb[ic]; tb[ic] = tm;
      }
    }
  }
  // Put the middle entry to its right place.
  tm = tb[ic]; tb[ic] = tb[min]; tb[min] = tm;

  if (ic-min<QSORT_LIMIT) FDRef_SimplSort( tb, min, ic-1 );
  else FDRef_QuickSort( tb, min, ic-1 );
  if (max-ic<QSORT_LIMIT) FDRef_SimplSort( tb, ic+1, max );
  else FDRef_QuickSort( tb, ic+1, max );
} // void SrvDir::FDRef_QuickSort( FDRef *tb, int min, int max ).



void SrvDir::FDRef_Sort( FDRef ** tb, int min, int max )
{
    if (max - min > QSORT_LIMIT) FDRef_QuickSort( tb, min, max );
    else FDRef_SimplSort( tb, min, max );
} // void SrvDir::FDRef_Sort( FDRef ** tb, int min, int max ).



void SrvDir::Sext_NCase() {
// When the flag Dir_NCase is set the list of extensions ...
// ... must be in Minor case.
//
    int       ii,   jj;
    char         * str;
    char            ch;

    ii = 0;
    while ((str = typls_[ii])) {
        jj = 0;
        while ((ch = str[jj])) {
            if (ch >= 'A' && ch <= 'Z') str[jj] += MaMi;
            jj++;
        }
        ii++;
    }
} // void SrvDir::Sext_NCase().



SrvDir::DNAcc SrvDir::Scan_Dir( FDAcc dref )
{
    DIR                  * dir; //  Directory file descriptor used by opendir(), ...
                                // ... readir() and closedir().
    struct dirent        * dsc; // Directory entry descriptor - from readir().
    struct stat          entbf; // File or Sub-directory descriptor - from stat().
    FDAcc                  ref, // The current File or Directory reference.
                         * dtb, // Directory table origine in dirstk_.
                         * ftb; // File table origine in filstk_.

    FDAcc               * ftab; // Pointer to the table of file in the current directory.
    DNAcc               * dtab, // Pointer to the table of sub_directories.
                          rdir; // Pointer to the resulting directory node.

    int    ndir,  nfil,    knd, // Found numbers of files and directories and,
          cndir, cnfil, ii, jj; // ... there cumulative counts, Service integers.

    int   cont_,  nerr,   ierr; // Flag for stop directory scan when 0, error codes.

    ierr = 0;

    // Open the directory.
#ifdef _HAVE_FL_UTF8_HDR_
    if (!(dir=opendir( path_.Path() )))
#else
    if (!(dir=opendir( path_.Path() )))
#endif
    {   // Error to open the directory.
        ierr = Error_Manager( -2, "Cannot open the directory :\n\tpath = \"%s\",\n\tSysMsg : %s\n",
                                  path_.Path(), strerror( errno ) );
        ierr = -2; return 0;                    // ... to stop scan at all nesting level.
    }

    // Open is a success.
    ierr = StartDir( path_.LastName() );        // Call the user routine for start directory scan.
    if (ierr < 0) {                             // If this routine send a stop scan.
        closedir( dir );                        //
        return 0;
    }
    nlvl_++;                                    // We update the nesting level.
    path_.Set( 0, 1 );                          // Append a directory separator to path.

    ndir = nfil = nerr = 0;                     // Init the count of directories, files and error(s).

    errno = 0; cont_ = 1;                       // clear errno to detect when we have a readdir error.
    while (cont_ && ierr >= 0) {
        dsc = readdir( dir );
        if (!dsc) { // We reach the end of directory or read error.
            cont_ = 0;
            if (errno) {                        // For error,
                ierr = Error_Manager( -2, "Read directory error :\n\tpath = \"%s\",\n\tSysMsg : %s\n",
                                  path_.Path(), strerror( errno ) );
                break;                          // ... to stop scan at all nesting level.
            }
        }

        if (!Check_FName(dsc->d_name)) break;   // Ignore any special entry.
        path_.Set( dsc->d_name, 0 );            // Append the entry name to path.

#if defined( _WIN32 ) && !defined( _CYGWIN_ )
        // On windows, no lstat (that does not follows the symbolic links.)
        if (stat( path_.Path(), &entbf ))   // Get the entry kind.
#else
        // ... else with Linux we can follow the symbolic link if the flag Sir_Slink is set
        if (TstFlg( Dir_SLink )) ierr = stat( path_.Path(), &entbf );
        // ... else we use lstat that read symbolic link as a small file that can be ignored.
        else ierr = lstat( path_.Path(), &entbf );
        if (ierr)
#endif
        { // On stat/lstat error.
            ierr = Error_Manager( 3, "%s\n\ttpath = \"%s\", \n\tSysMsg : %s\n",
                                     "(l)stat error on directory entry :",
                                     path_.Path(), strerror( errno )
                                );
            if (ierr < 0) break;
        }

        switch(entbf.st_mode & S_IFMT) {// Select the kind of directory entry.
            case S_IFDIR: { // For directories we restrick to folders with only one link.
                if (entbf.st_nlink == 1 && TstFlg( Dir_MLink ) ) {
                    if ((knd = Filter( 0, dsc->d_name )) >= 0) {
                        if (knd > 127) {
                            filtab_.AddObj( CreateRef( 1, knd-128, dsc->d_name ) );
                            nfil++;
                        } else {
                            dirtab_.AddObj( CreateRef( 1, knd, dsc->d_name ) );
                            ndir++;
                        }
                    }
                }
            }
            break;

            case S_IFREG: { // For regular file.
                // Check for file extension is matching.
                if ((knd = Filter( 1, dsc->d_name )) >= 0) {
                    filtab_.AddObj( CreateRef( 0, knd, dsc->d_name ) );
                    nfil++;
                }
            }
            break;

            default: ; // Ignored entry.
        }
    }
    closedir( dir );

    // When nothing was found in this directory we return NULL.
    if (!(ndir || nfil)) { nlvl_--; return 0; }

    dtb = disp_ - ndir;
    ftb = fisp_ - nfil;


    // Perform the sortings of entry when required.
    if (ndir > 1) FDRef_Sort( dtb, 0, ndir-1 );
    if (nfil > 1) FDRef_Sort( ftb, 0, nfil-1 );

    if (nfil) {
        ftab = new FDAcc[nfil];
        // For each File, call the Virtual SelFile function.
        for(ii = 0; ii < nfil; ii++) ftab[ii] = filtab_.Pop();
    } else ftab = 0;

    if (ndir) {
        jj = 0;
        dtab = 0;
        for(ii = 0; ii < ndir; ii++) {
            ref = dirtab_.Pop();
            path_.Set( ref->nam_, 0 );
            rdir = Scan_Dir( dirtab_.Pop() );
            if (rdir) {
                dintab_.Push( rdir );
                if (!jj) dtab = dnsp_;
                cndir += rdir->ndir_;
                cnfil += rdir->nfil_;
                jj++;
            }
            path_.Rem();
        }
        ndir = jj; // Update the sub-directory count (remove sub-dir without file selection).
    } else dtab = 0;

    nlvl_--;

    if (ndir || nfil) rdir = CreateDNode( dref, ndir, nfil, cndir, cnfil, ftab, dtab );
    else rdir = 0;

    return rdir;
} //SrvDir::DNAcc * SrvDir::Scan_Dir( FDAcc dref ).



int  SrvDir::BuildTree( const char * dn ) {
    FDAcc  ref;
    int     ie;

    path_.Set( dn, 0 ); // Set and fixe the base directory level and path.
    nlvl_ = -1;
    gbl_ierr = 0;       // Assume no error until showh otherwise.

    ccnfil_ = ccndir_ = 0;

    // When the default file filter (that check for extensions only) is used, ...
    // ... we must translate in minor case all specified extensions.
    //
    if (typls_ && TstFlg( Dir_NCase )) Sext_NCase();

    ref = CreateRef( 0, 0, dn );

    root_ = Scan_Dir( ref );   // We perform the scan of the tree.

    if (ie < 0) gbl_ierr = ie;
    else if (gbl_ierr < ie) gbl_ierr = ie;
    path_.Clear();
    nlvl_ =-1;
    return gbl_ierr;
} // int  SrvDir::Scan( const char * dn ).


//
// ------------------------------------------------------------------------------
//

//
// Main for Test (defined if DIR_DEBUG) is defined with compilation command :
//

#ifdef DIR_DEBUG

#define RECURSE_MAX  16;


SrvDir     sca_dir;
int     ncol =   1;

        
static void PrtSpace( int n ) {
    for (int ii=0; ii<n; ii++) printf( " " );
} // static void PrtSpace( int n ).



//
// Our specific routines called by sca_dir.Scan() :
//
static int Err_Handler( int ie, const char * msg ) {
    
    printf( "\n\n *** Our handler called with the following message\n" );
    printf( msg ); printf( "\n\n" );
    return ie;
} // static int Err_Handler( int ie, const char * msg ).



static int DirBegin( const char * nam ) {
    PrtSpace( ncol ); printf( " *BD1* We enter in the directory %s of deep = %d\n",
                              nam, sca_dir.NestLevel() );
    PrtSpace( ncol ); printf( " *BD2* complete path is now \"%s\"\n", sca_dir.Path() );
    ncol += 2;
    return 0;
} // static int DirBegin( const char * nam ).



static int DirEnd( int nd, int nf ) {
    ncol -= 2;
    PrtSpace( ncol ); printf( " *ED1* We have find %d directories and %d selected files\n", nd, nf );
    PrtSpace( ncol ); printf ( " *ED2* We return to previous directory at the deep %d\n",
                               sca_dir.NestLevel()
                             );
    return 0;
} // static int DirEnd( int nd, int nf ).



static int RegFile( int ty, const char * nam ) {
    PrtSpace( ncol ); printf( " -> we select the file \"%s\n", nam );
    return 0;
} // static int RegFile( int ty, const char * nam ).



// g++ -DDIR_DEBUG -o dirscan Service_DIR.cxx Service_Uti.cxx
//
// use with the command  "./dirscan [-f<flags>] <base_dir> <t1> <t2> t3> ...
//
// ... were flags 0 the default, 1 = Dir_MLink, 2 = Dir_SLink, 4 = Dir_NCase ...
// ... (or add of each one), and <t1>  <t2> <t3> ... the different selected file types.
//

int main( int argc, char ** argv ) {
    // Default we scan to look for these file extensions :
    const char * dxtb[] = { "jpeg", "JPEG", "jpg", "JPG", "png", "PNG", "bmp", "BMP", 0 };

    int       ii = 0,     ie = 0;
    char           * arg  =    0,
         * path = strdup( "./" );
    const   char *     tbext[65];
    int  hlp= 0, flg= 0, nar= -1;

    tbext[0] = 0;
    for(ii = 1; ii < argc; ii++) {
        arg = argv[ii];
        if (arg && arg[0]) {
            if (arg[0] == '-') {
                switch (arg[1]) {
                    case 'F': case 'f':
                        if (arg[2] >= '0' && arg[2] <= '7') flg = arg[2] - '0';
                    break;
                    case 'H': case 'h':
                        hlp = 1;
                        break;
                    default: ie = 1; // Unkown option.
                }
            } else {
                if (nar < 0) {
                    path = arg;
                    nar++;
                } else tbext[nar++] = arg;
            }
        }
    }

    if (nar <= 0) {
        nar = 0;
        while (dxtb[nar]) { tbext[nar] = dxtb[nar]; nar++; }
    }
    tbext[nar] = 0;
 
    if (ie || hlp) {
        printf( "    Use Service_DIR program by the command :\n\t %s %s\n\n",
                argv[0], "[-f<flags>] <base_dir> <t1> <t2> t3> ..." );
        printf( " Where :\n\t<flags> is 0, 1, 2 or 3 for Dir_MLink, Dir_SLink, Dir_MLink|Dir_SLink,\n" );
        printf( "\t<base_dir> is the root directory of the files tree to examine\n" );
        printf( "\t<t1>, <t2>, <t3> ... form the list of file types to lokk for.\n\n" );
        printf( " As result %s will print the list of path selected files\n\n", argv[0] );
        printf( " The option -h wil print this text.\n\n" );
    }

    printf( " We look for files with extensions in  %s :\n", path );
    ii = 0;
    while (tbext[ii]) {
        printf( "  %2d / %s\n", ii+1, tbext[ii] );
        ii++;
    }
    printf( " The scan will be performed with the flags [" );
    ii = 0;
    if (flg&SrvDir::Dir_MLink) { printf( "Dir_MLink" ); ii++; }
    if (flg&SrvDir::Dir_SLink) { if (ii) printf( "|" ); printf( "Dir_SLink" ); ii++; }
    if (flg&SrvDir::Dir_NCase) { if (ii) printf( "|" ); printf( "Dir_NCase" ); }

    printf( "]\n\n\n" );

    // Now we can work !
    if (!ie) {
        sca_dir.ErrHdl( Err_Handler );
        sca_dir.Setup( DirBegin,DirEnd, RegFile );
        sca_dir.SetFlg( flg );
        sca_dir.SetFltExt( (SrvDir::FTypLst) tbext );

        ie = sca_dir.Scan( path );
        printf( "\n\n the final scan return value is %d\n\n", ie );
    }
    return ie;
} // int main( int argc, char ** argv ).






#endif


//
// end of <PW-Id>.
//
