//
// <PW-Id>
//

///////////////////////////////////////////////////////////////////////////////
//                                                                           //
// Copyright 2018-2020 by Pierre Wolfers Meylan France                       //
//                                                                           //
// This library is free software. Distribution and use rights are outlined   //
// in the file "COPYING" which should have been included with this file.     //
// If this file is missing or damaged, see the license at:                   //
//                                                                           //
//    <To be define when ready>                                              //
//                                                                           //
//   This license described in this file overrides all other licenses that   //
//   might be specified in other files for this library.                     //
//                                                                           //
//   This library is free software; you can redistribute it  and/or modify   //
//   it under the terms of the GNU Lesser General Public License as publi-   //
//   shed by  the  Free Software  Foundation;  either  version 2.1  of the   //
//   License, or (at your option) any later version.                         //
//                                                                           //
//   This library  is  distributed in  the  hope that  it will  be useful,   //
//   but   WITHOUT  ANY  WARRANTY;  without even  the  implied warranty of   //
//   MERCHANTABILITY  or  FITNESS  FOR A  PARTICULAR PURPOSE.  See the GNU   //
//   Library General Public License for more details.                        //
//                                                                           //
//   You should have  received  a copy of  the  GNU Lesser General  Public   //
//   License  along with this library  (see COPYING.LIB); if not, write to   //
//   the Free Software Foundation :                                          //
//                        Inc., 675 Mass Ave, Cambridge, MA 02139, USA.      //
//                                                                           //
///////////////////////////////////////////////////////////////////////////////



//
// P.Wolfers Software
//

//
// Module of signal trap for debug routines/objects.
//


#include <stdio.h>
#include <errno.h>

#include "Service_TRAP.h"


static Cntx * stack_ = NULL;

Cntx::Cntx( const char * m, int l )
{
    prv_ = stack_;
    mdl_ =      m;
    lng_ =      l;
    stack_ = this;
} // Cntx::Cntx( const char * fnc, int lng ).


Cntx::~Cntx() { fprintf( stderr, " *** Rem from %s !!!\n", mdl_ ); stack_ = prv_; }

void Cntx::Line( int l ) { stack_->lng_ = l; }

void Cntx::Tracing( int nerr )
{
    Cntx * cnt = stack_;

    if (nerr) fprintf( stderr, "\n *** Detected Error # %d\n", nerr );
         else fprintf( stderr, "\n *** Marked Point\n" );
    if (cnt) {
        if (cnt->mdl_) fprintf( stderr, "\tin module %s", cnt->mdl_ );
        fprintf( stderr, " at ligne %d\n", cnt->lng_ );
        while (cnt = cnt->prv_) {
            fprintf( stderr, "\tcalled from " );
            if (cnt->mdl_) fprintf( stderr, "module %s ", cnt->mdl_ );
            fprintf( stderr, "at line %d\n", cnt->lng_ );
        }
    }
    fprintf( stderr, "\n" ); fflush( stderr );
} // void Cntx::Traceing().





//
//
//
// Management of signal/Interrupt.
//

void Error_Tracing( int );



/* Exception/Trap/Signal Manager */


#if defined( __WIN32 )&&!( __CYGWIN__ )

 /************************************************************
 *                                                           *
 *       Native Windows (using MINGW, without Cygwin)        *
 *                                                           *
 ************************************************************/

#   include <windows.h>
#   include <fenv.h>
#   include <float.h>

typedef short unsigned uint16_t;

///WIndows

/* Note : These routines are comming from :
 * https://stackoverflow.com/questions/30175524/enabling-floating-point-exceptions-on-mingw-gcc
 *
 */

static void feenableexcept( uint16_t fpflags )
{   /* edit 2015-12-17, my attempt at ASM code was giving me
     * problems in more complicated scenarios, so I
     * switched to using _controlfp_s. I finally posted it here
     * because of the upvote to the ASM version.
     *
     * {// http://stackoverflow.com/questions/247053/
     *      uint16_t mask(FE_ALL_EXCEPT & ~fpflags);
     *      asm("fldcw %0" : : "m" (mask) : "cc");
     * } //https://gcc.gnu.org/onlinedocs/gcc/Extended-Asm.html
     */
 
    unsigned int new_word = 0,
                       cw = 0;
    if (fpflags & FE_INVALID)   new_word |= _EM_INVALID;
    if (fpflags & FE_DIVBYZERO) new_word |= _EM_ZERODIVIDE;
    if (fpflags & FE_OVERFLOW)  new_word |= _EM_OVERFLOW;
    _controlfp_s(&cw,~new_word,_MCW_EM);
} // static void feenableexcept( uint16_t fpflags ).



static void fe_reset_traps()
{
    feclearexcept( FE_ALL_EXCEPT ); /* Clear x87 FE state. */
#ifdef __SSE__
    _MM_SET_EXCEPTION_STATE( 0 );   /* Clear SSE FE state. */
#endif
    feenableexcept( FE_DIVBYZERO|FE_OVERFLOW|FE_INVALID );  /* Set x87 FE State. */
#ifdef __SSE__
    _MM_SET_EXCEPTION_MASK( _MM_MASK_DENORM|_MM_MASK_UNDERFLOW|_MM_MASK_INEXACT );
#endif
} // static void fe_reset_traps();



/* Perhaps it is possible to use the windows native vectored exceptions */
static LONG WINAPI Trap_Handler( PEXCEPTION_POINTERS Except_Info )
{
  int ierr;

  /* Get the Exception information */
  PEXCEPTION_RECORD except = Except_Info->ExceptionRecord;
  PCONTEXT          contxt = Except_Info->ContextRecord;

  switch (except->ExceptionCode) {
    case EXCEPTION_IN_PAGE_ERROR:           ierr =  99; break;

    case EXCEPTION_ACCESS_VIOLATION:        ierr = -12; break;

    case EXCEPTION_STACK_OVERFLOW:
    case EXCEPTION_FLT_STACK_CHECK:         ierr =  99; break;

    case EXCEPTION_PRIV_INSTRUCTION:        ierr =  99; break;

    case EXCEPTION_ILLEGAL_INSTRUCTION:     ierr = -13; break;
 
    case EXCEPTION_INT_OVERFLOW:            ierr =  21; break;
    case EXCEPTION_INT_DIVIDE_BY_ZERO:      ierr =  22; break;

    case EXCEPTION_FLT_OVERFLOW:            ierr =  24; break;
    case EXCEPTION_FLT_DIVIDE_BY_ZERO:      ierr =  25; break;
    case EXCEPTION_FLT_UNDERFLOW:           ierr =  26; break;
    case EXCEPTION_FLT_INEXACT_RESULT:      ierr =  27; break;
    case EXCEPTION_FLT_DENORMAL_OPERAND:    ierr =  28; break;

    case EXCEPTION_ARRAY_BOUNDS_EXCEEDED:   ierr =  29; break;

    case EXCEPTION_FLT_INVALID_OPERATION:   ierr =  20; break;
      
    default:
      ierr = 10;
  }
  Error_Tracing( ierr );

  /* When the execution point is here, The error must be corrected */
  return EXCEPTION_CONTINUE_EXECUTION;
  /* return EXCEPTION_CONTINUE_SEARCH;  */
}


static LONG WINAPI Trap_Handler( PEXCEPTION_POINTERS Except_Info );



void Trap_Init()
{
    fe_reset_traps();
    AddVectoredExceptionHandler( 1, Trap_Handler );
} // void Trap_Init().




#elif (defined(_ARCH_PPC)&&defined(__APPLE__))

 /************************************************************
 *                                                           *
 *             Mac OS X on PPC (Apple Computer)              *
 *                                                           *
 ************************************************************/

#include <sys/stat.h>
#include <unistd.h>
#include <signal.h> 
#include <architecture/ppc/fp_regs.h>
#include <mach/mach.h>
#include <mach/mach_error.h>
#include <pthread.h>


struct _PPC_EnableFPEState {
  thread_t         targetThread;
  pthread_mutex_t  mutex;
  pthread_cond_t   condition;
  boolean_t        done;
};



static __private_extern__ void *_PPC_EnableFloatingPointExceptions(void *arg)
{
  // Set the MSR bits on how to handle the FPE exceptions.
  struct ppc_thread_state state;
  unsigned int  stateCount;
  kern_return_t krc;
  struct _PPC_EnableFPEState *q = (struct _PPC_EnableFPEState *)arg;

  stateCount = PPC_THREAD_STATE_COUNT;
  krc = thread_get_state(q->targetThread, PPC_THREAD_STATE, (natural_t *)&state, &stateCount);
  if (krc != KERN_SUCCESS) {
      mach_error("thread_get_state", krc);
      exit( 2 );
  }

#  define FE0_MASK (1<<11)
#  define FE1_MASK (1<<8)

  /* FE0  FE1
  //  0    0    -- Floating-point exceptions disabled
  //  0    1    -- Floating-point imprecise nonrecoverable
  //  1    0    -- Floating-point imprecise recoverable
  //  1    1    -- Floating-point precise mode

  //    fprintf(stderr, "state.srr1 = 0x%08x\n", state.srr1);
  //    fprintf(stderr, "FE0 = %d\n", (state.srr1 & FE0_MASK) != 0);
  //    fprintf(stderr, "FE1 = %d\n", (state.srr1 & FE1_MASK) != 0);
  */

  state.srr1 |= FE0_MASK;
  state.srr1 |= FE1_MASK;

  /*    fprintf(stderr, "state.srr1 = 0x%08x\n", state.srr1);
  //    fprintf(stderr, "FE0 = %d\n", (state.srr1 & FE0_MASK) != 0);
  //    fprintf(stderr, "FE1 = %d\n", (state.srr1 & FE1_MASK) != 0);
  */

  krc = thread_set_state(q->targetThread, PPC_THREAD_STATE, (natural_t *)&state, stateCount);
  if (krc != KERN_SUCCESS) {
    mach_error("thread_set_state", krc);
    exit( 2 );
  }

  if (0) {
    ppc_float_state_t floatState;
    ppc_fp_scr_t *fpscr;

    stateCount = PPC_FLOAT_STATE_COUNT;
    krc = thread_get_state(q->targetThread, PPC_FLOAT_STATE, (natural_t *)&floatState, &stateCount);
    if (krc != KERN_SUCCESS) {
      mach_error("thread_get_state", krc);
      exit( 2 );
    }

    /* Get a pointer through a bitfield type so we don't have to do index goo ourselves */
    fpscr = (ppc_fp_scr_t*)&floatState.fpscr_pad;
    PPC_PrintFPSCR(fpscr);
  }

  pthread_mutex_lock(&q->mutex);
  q->done = TRUE;
  pthread_cond_signal(&q->condition);
  pthread_mutex_unlock(&q->mutex);

  return NULL;
} // static __private_extern__ void *_PPC_EnableFloatingPointExceptions(void *arg).



static __private_extern__ void fpu_int_enable())
{ // PPC_EnableFloatingPointExceptions(
  ppc_fp_scr_t fpscr;

  /*  fprintf(stderr, "### Enabling FPU exceptions ###\n"); */
  fpscr = get_fp_scr();

  // Specify which exceptions we want
  fpscr.ve = 1;              /* invalid op exception enable */
  fpscr.oe = 1;              /* overflow exception enable */
  fpscr.ue = 0;              /* underflow exception enable */
  fpscr.ze = 1;              /* divide by zero exception enable */
  fpscr.xe = 0;              /* inexact exception enable */
  fpscr.ni = 1;              /* non-IEEE exception enable */

  /* Clear sticky exception bits so we don't immediately get clobbered with an exception */
  fpscr.fx = 0;              /* exception summary */
  fpscr.vx = 0;              /* invalid op exception summary */
  fpscr.ox = 0;              /* overflow exception */
  fpscr.ux = 0;              /* underflow exception */
  fpscr.zx = 0;              /* divide by zero exception */
  fpscr.xx = 0;              /* inexact exception */
  fpscr.vx_snan = 0;         /* not a number exception */
  fpscr.vx_isi = 0;          /* exception */
  fpscr.vx_idi = 0;          /* exception */
  fpscr.vx_zdz = 0;          /* exception */
  fpscr.vx_imz = 0;          /* exception */
  fpscr.vx_xvc = 0;          /* exception */
  fpscr.vx_soft = 0;         /* software request exception */
  fpscr.vx_cvi = 0;          /* invalid integer convert exception */

  /*
    fprintf(stderr, "FPSCR = 0x%08x : 0x%08x\n",
            ((unsigned int *)&fpscr)[0],
            ((unsigned int *)&fpscr)[1]);
  */

  set_fp_scr(fpscr);
  /*  fpscr = get_fp_scr(); */

  {
     struct _PPC_EnableFPEState state;
     pthread_t thread;

     memset(&state, 0, sizeof(state));
     state.targetThread = mach_thread_self();
     pthread_mutex_init(&state.mutex, NULL);
     pthread_cond_init(&state.condition, NULL);
        /* fprintf(stderr, "state.targetThread = 0x%08x\n", state.targetThread); */

     pthread_create(&thread, NULL, _PPC_EnableFloatingPointExceptions, &state);

     pthread_mutex_lock(&state.mutex);
     while (!state.done)
       pthread_cond_wait(&state.condition, &state.mutex);
     pthread_mutex_unlock(&state.mutex);
  }
} // static __private_extern__ void fpu_int_enable()).



static unsigned int fpu_state_reset()
{
  ppc_fp_scr_t fpscr;
  unsigned r = 0;

  fpscr = get_fp_scr();
/*  PPC_PrintFPSCR( &fpscr ); */

  if (fpscr.ox) r = FPE_FLTOVF_TRAP;
           else if (fpscr.zx) r = FPE_FLTDIV_TRAP;
                         else if (fpscr.vx) r = FPE_FLTINV_TRAP;

  fpscr.fx = 0;              /* exception summary */
  fpscr.vx = 0;              /* invalid op exception summary */
  fpscr.ox = 0;              /* overflow exception */
  fpscr.ux = 0;              /* underflow exception */
  fpscr.zx = 0;              /* divide by zero exception */
  fpscr.xx = 0;              /* inexact exception */
  fpscr.vx_snan = 0;         /* not a number exception */
  fpscr.vx_isi = 0;          /* exception */
  fpscr.vx_idi = 0;          /* exception */
  fpscr.vx_zdz = 0;          /* exception */
  fpscr.vx_imz = 0;          /* exception */
  fpscr.vx_xvc = 0;          /* exception */
  fpscr.vx_soft = 0;         /* software request exception */
  fpscr.vx_cvi = 0;          /* invalid integer convert exception */

  set_fp_scr( fpscr );

  return r;
} // static unsigned int fpu_state_reset().



static void Trap_Handler( int signum, siginfo_t * info, void * p )
{
  int ierr, status;

  ierr = 0;
  switch (signum)
  {
#  ifdef SIGBUS
    case SIGBUS:  ierr = -11;  break;
#  endif

#  ifdef SIGSEGV
    case SIGSEGV: ierr = -12;  break;
#  endif

#  ifdef SIGILL
    case SIGILL:  ierr = -13;  break;
#  endif

#  ifdef SIGFPE
    case SIGFPE:
      ierr = fpu_state_reset();
      switch (info->si_code)
      {
        case FPE_INTOVF: ierr = 21; break;
        case FPE_INTDIV: ierr = 22; break;
        case FPE_FLTOVF: ierr = 24; break;
        case FPE_FLTDIV: ierr = 25; break;
        case FPE_FLTUND: ierr = 26; break;
        case FPE_FLTRES: ierr = 27; break;
        case FPE_FLTINV: ierr = 28; break;
        case FPE_FLTSUB: ierr = 29; break;
        default:  ierr = 20;
      }
    break;
#  endif

    default:
      ierr = 10;
  }
  Error_Tracing( ierr );
} // static void Trap_Handler( int signum, siginfo_t * info, void * p ).



void Trap_Init()
{ /* Rev. of 02-JUN-2010 */
  struct sigaction action;

  action.sa_sigaction = Trap_Handler;
  sigemptyset( &action.sa_mask );
  action.sa_flags = SA_SIGINFO;

  fpu_int_enable();

#  ifdef SIGFPE
  sigaction( SIGFPE, &action, NULL );
#  endif

#  ifdef SIGBUS
  sigaction( SIGBUS, &action, NULL );
#  endif

#  ifdef SIGSEGV
  sigaction( SIGSEGV, &action, NULL );
#  endif

#  ifdef SIGILL
  sigaction( SIGILL, &action, NULL );
#  endif
} // void Trap_Init().



#elif (defined(__UNIX)&&defined(_SGI))

 /************************************************************
 *                                                           *
 *                Silicon Graphic Unix (IRIX)                *
 *                                                           *
 ************************************************************/

#  include <signal.h>
#  include <unistd.h>
#  include <math.h>
#  include <sys/fpu.h>


static void fpu_int_enable()
{ union fpc_csr f;
  f.fc_word=get_fpc_csr();
  f.fc_struct.en_divide0   = 1;
  f.fc_struct.en_underflow = 0;
  f.fc_struct.en_overflow  = 1;
  f.fc_struct.en_invalid   = 1;
  set_fpc_csr(f.fc_word);
  return;
}  static void fpu_int_enable().



static void Trap_Handler( int signum, siginfo_t * info, void * p )
{
  int ierr;

  ierr = 0;
  switch (signum)
  {
#  ifdef SIGBUS
    case SIGBUS:  ierr = -11;  break;
#  endif

#  ifdef SIGSEGV
    case SIGSEGV: ierr = -12;  break;
#  endif

#  ifdef SIGILL
    case SIGILL:  ierr = -13;  break;
#  endif

#  ifdef SIGFPE
    case SIGFPE:
/*    status = feclearexcept( FE_ALL_EXCEPT ); */
      switch (info->si_code)
      {
        case FPE_INTOVF: ierr = 21; break;
        case FPE_INTDIV: ierr = 22; break;
        case FPE_FLTOVF: ierr = 24; break;
        case FPE_FLTDIV: ierr = 25; break;
        case FPE_FLTUND: ierr = 26; break;
        case FPE_FLTRES: ierr = 27; break;
        case FPE_FLTINV: ierr = 28; break;
        case FPE_FLTSUB: ierr = 29; break;
        default:  ierr = 20;
      }
    break;
#  endif

    default:
      ierr = 10;
  }
  Error_Tracing( ierr );
} // static void Trap_Handler( int signum, siginfo_t * info, void * p ).



void Trap_Init()
{ /* Rev. of 02-JUN-2010 */
  struct sigaction action;

  action.sa_sigaction = Trap_Handler;
  sigemptyset( &action.sa_mask );
  action.sa_flags = SA_SIGINFO;

  fpu_int_enable();

#  ifdef SIGFPE
  sigaction( SIGFPE, &action, NULL );
#  endif

#  ifdef SIGBUS
  sigaction( SIGBUS, &action, NULL );
#  endif

#  ifdef SIGSEGV
  sigaction( SIGSEGV, &action, NULL );
#  endif

#  ifdef SIGILL
  sigaction( SIGILL, &action, NULL );
#  endif
} // void Trap_Init().




#elif (defined(__LINUX__)||defined(__LINUX)||defined(__CYGWIN__))


 /************************************************************
 *                                                           *
 *                 Linux  Operating Systems                  *
 *                                                           *
 ************************************************************/

#ifndef _GNU_SOURCE
# define _GNU_SOURCE
#endif

#include <sys/stat.h>
#include <signal.h>
#ifndef __USE_GNU 
#  define __USE_GNU
#endif
#include <fenv.h>
#include <unistd.h>
#include <math.h>



static void EnableFpu_Exception()
{
  feenableexcept( FE_DIVBYZERO | FE_INVALID | FE_OVERFLOW );
/*
  fexcept_t fpu_exc;
    fegetexceptflag( &fpu_exc,  FE_DIVBYZERO | FE_INVALID | FE_OVERFLOW );
    fesetexceptflag( &fpu_exc,  FE_DIVBYZERO | FE_INVALID | FE_OVERFLOW );
*/
} // static void EnableFpu_Exception().



/* Revision from of 02-JUN-2010 */
static void Trap_Handler( int signum, siginfo_t * info, void * p )
{
  int ierr, status;

  ierr = 0;
  switch (signum)
  {
#  ifdef SIGBUS
    case SIGBUS:  ierr = -11;  break;
#  endif

#  ifdef SIGSEGV
    case SIGSEGV: ierr = -12;  break;
#  endif

#  ifdef SIGILL
    case SIGILL:  ierr = -13;  break;
#  endif

#  ifdef SIGFPE
    case SIGFPE:
      EnableFpu_Exception();
      /* feenableexcept( FE_DIVBYZERO | FE_INVALID | FE_OVERFLOW ); */
      switch (info->si_code)
      {
        case FPE_INTOVF: ierr = 21; break;
        case FPE_INTDIV: ierr = 22; break;
        case FPE_FLTOVF: ierr = 24; break;
        case FPE_FLTDIV: ierr = 25; break;
        case FPE_FLTUND: ierr = 26; break;
        case FPE_FLTRES: ierr = 27; break;
        case FPE_FLTINV: ierr = 28; break;
        case FPE_FLTSUB: ierr = 29; break;
        default:  ierr = 20;
      }
    break;
#  endif

    default:
      ierr = 10;
  }
  Error_Tracing( ierr );
} // static void Trap_Handler( int signum, siginfo_t * info, void * p ).



void Trap_Init()
{ /* Rev. of 02-JUN-2010 */
  struct sigaction action;

  action.sa_sigaction = Trap_Handler;
  sigemptyset( &action.sa_mask );
  action.sa_flags = SA_SIGINFO;

  EnableFpu_Exception();

#  ifdef SIGFPE
  sigaction( SIGFPE, &action, NULL );
#  endif

#  ifdef SIGBUS
  sigaction( SIGBUS, &action, NULL );
#  endif

#  ifdef SIGSEGV
  sigaction( SIGSEGV, &action, NULL );
#  endif

#  ifdef SIGILL
  sigaction( SIGILL, &action, NULL );
#  endif
} // void Trap_Init().



#else
#  error " System not supported => Signal module is Empty."
#endif








typedef struct errhdl_rec *ptr_errhdl;

typedef int (*errhdl)( int );

struct errhdl_rec { ptr_errhdl prv;
                    errhdl     fnc;
                  };

static int  ncnt  =  0; // Error loop counter */

static ptr_errhdl errhdl_heap = NULL;


static int err_tab[] = {
       /* (VMS spc., error code,)..... */

#      ifdef EVMSERR
         EVMSERR, 200,         /* Not OWNER */
#      endif

#      ifdef ENOENT
         ENOENT,  202,         /* No such file or directory */
#      endif

#      ifdef EIO
         EIO,     203,         /* IO error */
#      endif

#      ifdef ENXIO
         ENXIO,   204,         /* No such device or address */
#      endif

#      ifdef EBADF
         EBADF,   205,         /* Bad file number */
#      endif

#      ifdef EACCES
         EACCES,  206,         /* Permission denied */
#      endif

#      ifdef ENOTBLK
         ENOTBLK, 207,         /* Block device required */
#      endif

#      ifdef EBUSY
         EBUSY,   208,         /* Mount devices busy */
#      endif

#      ifdef EEXIST
         EEXIST,  209,         /* File already exist */
#      endif

#      ifdef EXDEV
         EXDEV,   210,         /* Cross-device link */
#      endif

#      ifdef ENODEV
         ENODEV,  211,         /* No such device */
#      endif

#      ifdef ENOTDIR
         ENOTDIR, 212,         /* Not a directory */
#      endif

#      ifdef EISDIR
         EISDIR,  213,         /* Bad use of a directory */
#      endif

#      ifdef ENFILE
         ENFILE,  214,         /* File table overflow */
#      endif

#      ifdef EMFIL
         EMFIL,   215,         /* Too many open files */
#      endif

#      ifdef ENOTTY
         ENOTTY,  216,         /* Not a typewriter */
#      endif

#      ifdef ETXTBSY
         ETXTBSY, 217,         /* Text file busy */
#      endif

#      ifdef EFBIG
         EFBIG,   218,         /* File too big */
#      endif

#      ifdef ENOSPC
         ENOSPC,  219,         /* No space left on device */
#      endif

#      ifdef ESPIPE
         ESPIPE,  220,         /* Illegal seek */
#      endif

#      ifdef EROFS
         EROFS,   221,         /* Read only file system */
#      endif

#      ifdef EMLINK
         EMLINK,  222,         /* Too many links */
#      endif

#      ifdef EPIPE
         EPIPE,   223,         /* Broken pipe */
#      endif

#      ifdef EWOULDBLOCK
         EWOULDBLOCK, 224,     /* File I/O buffer are empty */
#      endif

#      ifdef ESRCH
         ESRCH,   301,         /* No such process */
#      endif

#      ifdef EINTR
         EINTR,   401,         /* Interrupt system call */
#      endif

         0,        10          /* UNKNOWN ERROR */
  };





int CC_Error()
{
  int i;

  i = 0;
  while ((err_tab[i++] != errno) && (err_tab[i-1])) i++;
  return err_tab[i];
} // int CC_Error().



void Establish_Handler( errhdl f )
{
  ptr_errhdl p;

  p = (ptr_errhdl) malloc( sizeof( struct errhdl_rec ) );
  p->prv  = errhdl_heap;
  p->fnc  = f;
  errhdl_heap = p;
} // Establish_Handler( errhdl f ).



void Revert_Handler()
{
  ptr_errhdl p;
  p = errhdl_heap;
  errhdl_heap = p->prv;
  free( p );
} // Revert_Handler().


typedef struct CNTX_REC * CntxPtr;

typedef struct CNTX_REC {
    CntxPtr       prv;
    const char *  fnc;
    int           lng;
} CntxRec;



/*
void Gen_Error_Msg( int nerr )
{
    CntxPtr cnt = Cntx_Stack;

    fprintf( stderr, "\n *** Detected Error # %d\n", nerr );
    if (cnt) {
        if (cnt->fnc) fprintf( stderr, "\tin module %s", cnt->fnc );
        fprintf( stderr, " at ligne %d\n", cnt->lng );
        while (cnt = cnt->prv) {
            fprintf( stderr, "\tcalled from " );
            if (cnt->fnc) fprintf( stderr, "module %s ", cnt->fnc );
            fprintf( stderr, "at line %d\n", cnt->lng );
        }
    }
    fprintf( stderr, "\n" ); fflush( stderr );
} // void Gen_Error_Msg( int nerr ).
*/


void Error_Tracing( int nerr )
{ ptr_errhdl p;
  int        crd;
  /* Negative error code => Fatal, the program do not must continue */

  nerr = abs( nerr );

  if (ncnt>=10) exit(4);

  ncnt++;

  /* If some user error handlers are set => call it */
  p = errhdl_heap;
  crd = 0;
  while ((p != NULL) && (crd == 0))
  { crd = p->fnc( nerr );
    p = p->prv;
  }
  if (crd <= 0)
  { /* Standard action : message and exit */
    stack_->Tracing();
//  Gen_Error_Msg( abs( nerr ) );

    exit(2);
  }
  ncnt--;
} // Error_Tracing( int nerr ).



//
// end of <PW-Id>.
//
